package ru.yandex.travel.orders.workflows.order.hotel;

import java.math.BigDecimal;
import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.TMoneyMarkup;
import ru.yandex.travel.orders.workflow.order.proto.TStartInvoiceRefund;
import ru.yandex.travel.orders.workflow.order.proto.TStartMoneyOnlyRefund;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.orders.workflows.orderitem.TargetFiscalItems;
import ru.yandex.travel.workflow.StateContext;

@Slf4j
public class MoneyRefundUtils {
    public static void startMoneyOnlyRefund(TStartMoneyOnlyRefund event,
                                            StateContext<EHotelOrderState, HotelOrder> context,
                                            FinancialEventService financialEventService) {
        HotelOrder order = context.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1, "Can't refund money with more than 1 order item");
        Preconditions.checkState(order.getPendingMoneyRefund() == null, "Other Money Refund is in progress");
        HotelOrderItem hotelOrderItem = (HotelOrderItem)order.getOrderItems().get(0);
        Preconditions.checkState(event.getRefundUserMoney() || event.getGenerateFinEvents(),
                "At least one of RefundUserMoney or GenerateFinEvents should be set");

        Money newFiscalAmount = ProtoUtils.fromTPrice(event.getNewInvoiceAmount());
        MoneyMarkup newFiscalAmountMarkup = fromTMoneyMarkup(event.getNewInvoiceAmountMarkup());
        Money refundAmount = ProtoUtils.fromTPrice(event.getRefundAmount());
        Money currentOrderMoney = getCurrentOrderMoney(hotelOrderItem);
        Preconditions.checkState(newFiscalAmount.isPositiveOrZero(),
                "newFiscalAmount (%s) should be positive or zero", newFiscalAmount);
        Preconditions.checkState(refundAmount.isPositive(),
                "refundAmount (%s) should be positive", refundAmount);
        Preconditions.checkState(newFiscalAmount.add(refundAmount).isEqualTo(currentOrderMoney),
                "newFiscalAmount (%s) + refundAmount (%s) should be equal to current order money amount (%s)",
                newFiscalAmount, refundAmount, currentOrderMoney);

        order.setUserActionScheduled(false);

        if (event.getRefundUserMoney()) {
            TargetFiscalItems targetFiscalItems = RefundingUtils.calculateTargetFiscalItemsGroupingByType(
                    hotelOrderItem.getFiscalItems(), newFiscalAmount, newFiscalAmountMarkup);

            MoneyRefund refund = MoneyRefund.createPendingRefund(context.getWorkflowEntity(),
                    targetFiscalItems.getPrices(), targetFiscalItems.getPricesMarkup(), null, "Cancelling the order");
            refund.getContext().setTargetHotelOrderState(context.getState());
            refund.setState(MoneyRefundState.IN_PROGRESS);

            context.setState(EHotelOrderState.OS_WAITING_MONEY_ONLY_REFUND);
            context.scheduleEvent(TStartInvoiceRefund.newBuilder()
                    .setReason(event.getReason())
                    .putAllTargetFiscalItems(RefundingUtils.convertTargetFiscalItemsToProto(
                            targetFiscalItems.getPrices()))
                    .putAllTargetFiscalItemsMarkup(RefundingUtils.convertTargetFiscalItemsMarkupToProto(
                            targetFiscalItems.getPricesMarkup()))
                    .build());
        } else {
            log.warn("Trust refund is skipped on money-only refund");
        }

        if (event.getGenerateFinEvents()) {
            financialEventService.registerRefundedService(hotelOrderItem, event.getMoneyRefundMode(), newFiscalAmount, refundAmount, Instant.now());
        } else {
            log.warn("Generation of financial events is skipped on money-only refund");
        }
    }

    public static Money getCurrentOrderMoney(HotelOrderItem hotelOrderItem) {
        if (hotelOrderItem.isPostPaid())
            return hotelOrderItem.getHotelItinerary().getRealHotelPrice();
        //noinspection SuspiciousMethodCalls
        return hotelOrderItem.getOrder().getInvoices().stream()
                .filter(i -> !TrustInvoice.UNPAID_INVOICE_STATES.contains(i.getInvoiceState()))
                .map(Invoice::calculateCurrentAmount)
                .reduce(BigDecimal::add)
                .map(x -> Money.of(x, hotelOrderItem.getOrder().getCurrency()))
                .orElse(Money.zero(hotelOrderItem.getOrder().getCurrency()));
    }

    public static MoneyMarkup fromTMoneyMarkup(TMoneyMarkup markup) {
        // the first param doesn't actually matter as we do a less strict value check
        return fromTMoneyMarkup(true, markup);
    }

    public static MoneyMarkup fromTMoneyMarkup(boolean isSet, TMoneyMarkup markup) {
        TMoneyMarkup defaultValue = TMoneyMarkup.getDefaultInstance();
        if (!isSet || markup == null || markup.equals(defaultValue) || markup.getCard().equals(TPrice.getDefaultInstance())) {
            // is not set in a proper way
            return null;
        }
        Preconditions.checkState(markup.hasCard() && markup.hasYandexAccount(),
                "The card and yandex account values have to be explicitly set; markup %s", markup);
        return MoneyMarkup.builder()
                .card(ProtoUtils.fromTPrice(markup.getCard()))
                .yandexAccount(ProtoUtils.fromTPrice(markup.getYandexAccount()))
                .build();
    }
}
