package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.time.Instant;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundContext;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.TScheduleClearing;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TStartInvoiceRefund;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@RequiredArgsConstructor
@Slf4j
public abstract class BaseWaitingMoneyRefundStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState, HotelOrder> {

    @HandleEvent
    public void handleVoucherCreated(TVoucherCreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelHandlerCommon.handleDocumentCreated(event.getVoucherType(), event.getVoucherUrl(), context);
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherRecreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelHandlerCommon.handleDocumentCreated(event.getVoucherType(), event.getVoucherUrl(), context);
    }


    @HandleEvent
    public void handleStartInvoiceRefund(TStartInvoiceRefund event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1,
                "Can't refund money for order with multiple order items");
        MoneyRefund refund = order.getInProgressMoneyRefund();
        if (refund.getContext().getInvoicesToRefund().isEmpty()) {
            finishRefund(order.getInProgressMoneyRefund(), ctx);
            log.info("No invoices to refund, confirming the refund process without ever going to trust");
        } else {
            Map<UUID, Invoice> invoicesById = order.getInvoices().stream().collect(Collectors.toMap(Invoice::getId, Function.identity()));
            refund.getContext().getInvoicesToRefund().values()
                    .forEach(invoiceRefund -> {
                        Preconditions.checkState(invoicesById.containsKey(invoiceRefund.getInvoiceId()),
                                "Invoice to refund does not belong to order");
                        ctx.scheduleExternalEvent(invoicesById.get(invoiceRefund.getInvoiceId()).getWorkflow().getId(),
                                RefundingUtils.createRefundEvent(invoiceRefund.getTargetPricesByFiscalItem(),
                                        invoiceRefund.getTargetPricesByFiscalItemMarkup(), getMoneyRefundReason(event)));
                    });
        }
    }

    @HandleEvent
    public void handleInvoiceRefunded(TInvoiceRefunded event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder hotelOrder = ctx.getWorkflowEntity();
        MoneyRefund refund = hotelOrder.getInProgressMoneyRefund();
        UUID invoiceId = UUID.fromString(event.getInvoiceId());
        if (refund.getContext().getInvoicesToRefund().containsKey(invoiceId)) {
            refund.getContext().getInvoicesToRefund().get(invoiceId).setState(MoneyRefundContext.InvoiceRefundState.REFUNDED);
        }
        var nonRefundedInvoices = refund.getContext().getInvoicesToRefund().values().stream()
                .filter(ir -> ir.getState() != MoneyRefundContext.InvoiceRefundState.REFUNDED)
                .map(ir -> ir.getInvoiceId().toString())
                .collect(Collectors.toList());
        if (nonRefundedInvoices.size() > 0) {
            log.info("Invoices {} are not refunded yet", StringUtils.join(nonRefundedInvoices, ", "));
            return;
        }

        refund.getContext().getInvoicesToRefund().values()
                .forEach(ir -> {
                    Invoice invoice = hotelOrder.getInvoices().stream()
                            .filter(i -> i.getId().equals(ir.getInvoiceId()))
                            .findFirst().orElseThrow();
                    ctx.scheduleExternalEvent(invoice.getWorkflow().getId(),
                            TScheduleClearing.newBuilder().setClearAt(ProtoUtils.fromInstant(Instant.now())).build());
                });

        finishRefund(refund, ctx);
    }

    protected abstract void finishRefund(MoneyRefund refund, StateContext<EHotelOrderState, HotelOrder> ctx);

    protected abstract String getMoneyRefundReason(TStartInvoiceRefund event);
}
