package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.proto.ERefundReason;
import ru.yandex.travel.orders.workflow.hotels.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TRenderDocuments;
import ru.yandex.travel.orders.workflow.order.proto.TStartManualServiceRefund;
import ru.yandex.travel.orders.workflow.order.proto.TStartMoneyOnlyRefund;
import ru.yandex.travel.orders.workflow.order.proto.TStartServiceRefund;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherType;
import ru.yandex.travel.orders.workflow.voucher.proto.TGenerateVoucher;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.order.hotel.MoneyRefundUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Slf4j
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TClearingInProcess.class,
        TInvoiceCleared.class,
})
@RequiredArgsConstructor
public class ConfirmedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState, HotelOrder> {
    private final VoucherRepository voucherRepository;
    private final WorkflowRepository workflowRepository;
    private final NotificationHelper notificationHelper;
    private final RefundCalculationService refundCalculationService;
    private final FinancialEventService financialEventService;

    @HandleEvent
    public void handleRenderDocuments(TRenderDocuments event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelOrder order = context.getWorkflowEntity();
        Voucher voucher = Voucher.createForOrder(order);
        voucher = voucherRepository.saveAndFlush(voucher);
        Workflow voucherWorkflow = Workflow.createWorkflowForEntity(voucher,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        voucherWorkflow = workflowRepository.saveAndFlush(voucherWorkflow);

        context.scheduleExternalEvent(voucherWorkflow.getId(), TGenerateVoucher.newBuilder().build());
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherCreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        EVoucherType docType = event.getVoucherType();
        HotelHandlerCommon.handleDocumentCreated(docType, event.getVoucherUrl(), context);

        if (docType == EVoucherType.VT_HOTELS || docType == EVoucherType.VT_UNKNOWN) {
            HotelOrder order = context.getWorkflowEntity();
            UUID notificationWorkflowId = notificationHelper.createWorkflowForSuccessfulHotelNotification(order);
            var senderRequest = TSend.newBuilder().build();
            context.scheduleExternalEvent(notificationWorkflowId, senderRequest);
        }
    }

    @HandleEvent
    public void handleVoucherRecreated(TVoucherRecreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelHandlerCommon.handleDocumentCreated(event.getVoucherType(), event.getVoucherUrl(), context);
    }

    @HandleEvent
    public void handleNotificationComplete(TNotificationComplete event,
                                           StateContext<EHotelOrderState, HotelOrder> context) {
        log.info("Confirmation notification sent for order {}", context.getWorkflowEntity().getId());
    }

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<EHotelOrderState, HotelOrder> context) {
        log.info("Money acquired from extra/deferred payment {} for confirmed order", event.getPaymentId());
        HotelOrder order = context.getWorkflowEntity();
        if (order.getDocumentUrl() != null) {
            log.info("Re-sending notification email");
            UUID notificationWorkflowId = notificationHelper.createWorkflowForSuccessfulHotelNotification(order);
            var senderRequest = TSend.newBuilder().build();
            context.scheduleExternalEvent(notificationWorkflowId, senderRequest);
        } else {
            log.warn("No voucher yet. It will be sent once created");
        }
        for (OrderItem item : order.getOrderItems()) {
            context.scheduleExternalEvent(item.getWorkflow().getId(), event.toBuilder().build());
        }
    }

    @HandleEvent
    public void handleStartRefund(TStartServiceRefund event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelOrder order = context.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1, "Can't refund order with more than 1 order item");
        order.getPayments().forEach(p ->
                context.scheduleExternalEvent(p.getWorkflow().getId(), TCancelPayment.newBuilder().build()));

        order.setUserActionScheduled(false);
        context.scheduleExternalEvent(order.getOrderItems().get(0).getWorkflow().getId(),
                TRefundStart.newBuilder().setToken(event.getToken())
                        .setReason(ERefundReason.RR_USER)
                        .build());
        context.setState(EHotelOrderState.OS_WAITING_SERVICE_REFUND);
    }

    @HandleEvent(TStartManualServiceRefund.class)
    public void handleStartManualRefund(TStartManualServiceRefund event,
                                        StateContext<EHotelOrderState, HotelOrder> context) {
        HotelOrder order = context.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1, "Can't manual refund order with more than 1 order" +
                " item");
        order.setUserActionScheduled(false);
        order.getPayments().forEach(p ->
                context.scheduleExternalEvent(p.getWorkflow().getId(), TCancelPayment.newBuilder().build()));
        context.scheduleExternalEvent(order.getOrderItems().get(0).getWorkflow().getId(),
                TRefundStart.newBuilder()
                        .setToken(event.getToken())
                        .setSkipFinEvents(event.getSkipFinEvents())
                        .setMoneyRefundMode(event.getMoneyRefundMode())
                        .setForce(true)
                        .setReason(ERefundReason.RR_OPERATOR)
                        .setRefundDescription(event.getRefundDescription())
                        .build());
        context.setState(EHotelOrderState.OS_WAITING_SERVICE_REFUND);
    }

    @HandleEvent
    public void handleStartMoneyOnlyRefund(TStartMoneyOnlyRefund event,
                                           StateContext<EHotelOrderState, HotelOrder> context) {
        MoneyRefundUtils.startMoneyOnlyRefund(event, context, financialEventService);
    }

    @HandleEvent
    public void handlePaymentCancelled(TPaymentCancelled cancelled,
                                       StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getPaymentSchedule() != null, "No payment schedule");
        Preconditions.checkState(order.getPaymentSchedule().getState() == EPaymentState.PS_CANCELLED,
                "Unexpected state of payment schedule: " + order.getPaymentSchedule().getState());
        Preconditions.checkState(order.getOrderItems().size() == 1,
                "Unexpected amount of order items");
        log.info("Payment schedule was cancelled while the order was cancelled: " +
                "the deferred payment was not made in time, so refunding the order");

        var refundCalculationWrapper = refundCalculationService.calculateRefundForHotelItemFromRules(
                (HotelOrderItem) order.getOrderItems().get(0),
                order.getCurrency());
        log.info("Will refund using the following refund calculation: {}",
                refundCalculationWrapper.getRefundCalculation());
        ctx.scheduleExternalEvent(order.getOrderItems().get(0).getWorkflow().getId(),
                TRefundStart.newBuilder()
                        .setToken(refundCalculationWrapper.getRefundToken())
                        .setReason(ERefundReason.RR_SCHEDULE)
                        .build());
        ctx.setState(EHotelOrderState.OS_WAITING_SERVICE_REFUND);
    }
}
