package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TClearingInProcess.class,
        TInvoiceCleared.class,
        TVoucherCreated.class,
        TVoucherRecreated.class,
})
@RequiredArgsConstructor
public class ManualProcessingStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState, HotelOrder> {
    private final CheckMoneyRefundsService checkMoneyRefundsService;

    @HandleEvent
    public void handleManualRefund(TManualRefund event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        var hotelOrder = ctx.getWorkflowEntity();
        if (checkMoneyRefundsService.checkManualRefundAllowed(hotelOrder)) {
            MoneyRefund moneyRefund = MoneyRefund.createPendingRefundFromProto(hotelOrder,
                    event.getTargetFiscalItemsMap(), event.getTargetFiscalItemsMarkupMap(), null, event.getReason());
            moneyRefund.setState(MoneyRefundState.IN_PROGRESS);

            Invoice invoice = hotelOrder.getCurrentInvoice();
            Preconditions.checkState(invoice != null, "Current invoice must be present in order");

            ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), RefundingUtils.createRefundEvent(
                    moneyRefund.getTargetFiscalItems(), moneyRefund.getTargetFiscalItemsMarkup(),
                    moneyRefund.getReason()));
        } else {
            log.warn("Money refund not allowed in state {} for order {}", hotelOrder.getState(), hotelOrder.getId());
            hotelOrder.setUserActionScheduled(false);
        }
    }

    @HandleEvent
    public void handleInvoiceRefunded(TInvoiceRefunded event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(ctx.getWorkflowEntity());
        Preconditions.checkState(invoice.getId().equals(UUID.fromString(event.getInvoiceId())), "Invoice with id %s " +
                        "must be current",
                event.getInvoiceId());

        MoneyRefund refund = ctx.getWorkflowEntity().getInProgressMoneyRefund();
        Preconditions.checkNotNull(refund, "No money refund in progress is available");
        refund.setState(MoneyRefundState.REFUNDED);
        ctx.getWorkflowEntity().setUserActionScheduled(false);
    }
}
