package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TStartMoneyOnlyRefund;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.order.hotel.MoneyRefundUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TClearingInProcess.class,
        TInvoiceCleared.class,
        TVoucherCreated.class,
        TVoucherRecreated.class,
        TPaymentCancelled.class,
})
@RequiredArgsConstructor
public class RefundStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState, HotelOrder> {
    private final CheckMoneyRefundsService checkMoneyRefundsService;
    private final FinancialEventService financialEventService;

    @HandleEvent
    public void handleNotificationComplete(TNotificationComplete event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        log.info("Refund notification sent for order " + ctx.getWorkflowEntity().getId());
    }

    @HandleEvent
    public void handleManualRefund(TManualRefund event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        if (checkMoneyRefundsService.checkManualRefundAllowed(order)) {
            order.setState(EHotelOrderState.OS_MANUAL_PROCESSING);
            ctx.scheduleEvent(event);
        } else {
            log.warn("Money refund not allowed in state {} for order {}", order.getState(), order.getId());
            order.toggleUserActionScheduled(false);
        }
    }

    @HandleEvent
    public void handleStartMoneyOnlyRefund(TStartMoneyOnlyRefund event,
                                           StateContext<EHotelOrderState, HotelOrder> context) {
        MoneyRefundUtils.startMoneyOnlyRefund(event, context, financialEventService);
    }
}
