package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.util.Optional;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TClearingInProcess.class,
        TInvoiceCleared.class,
        TMoneyAcquired.class
})
public class WaitingCancellationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState,
        HotelOrder> {
    private final PromoCodeApplicationService promoCodeApplicationService;

    @HandleEvent
    public void handleInvoiceRefunded(TInvoiceRefunded event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        //TODO (mbobrov): think of checking several invoices, if there're some
        HotelOrder order = ctx.getWorkflowEntity();
        Optional<Invoice> invoice =
                order.getInvoices().stream().filter(i -> i.getId().equals(UUID.fromString(event.getInvoiceId())))
                        .findFirst();
        Preconditions.checkState(invoice.isPresent(), "Invoice with id %s must be present in order",
                event.getInvoiceId());
        Invoice inv = invoice.get();
        Preconditions.checkState(
                inv.getInvoiceState() == ETrustInvoiceState.IS_CANCELLED
                        || inv.getInvoiceState() == ETrustInvoiceState.IS_REFUNDED,
                "Invoice with id %s must be cancelled or fully refunded", event.getInvoiceId()
        );
        promoCodeApplicationService.freePromoCodeActivations(order);
        ctx.setState(EHotelOrderState.OS_CANCELLED);
    }

    @HandleEvent
    public void handlePaymentCancelled(TPaymentCancelled event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        checkAllServicesAndPaymentsAreCancelled(ctx);
    }

    @HandleEvent
    public void handleServiceCancelled(TServiceCancelled event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1, "Handling cancellation of multiple items is not " +
                "implemented");
        UUID orderItemId = ProtoChecks.checkStringIsUuid("order item id", event.getServiceId());
        OrderItem orderItem = order.getOrderItems().get(0);
        Preconditions.checkState(orderItem.getId().equals(orderItemId), "Got ServiceCancelled event for order item " +
                "%s, but order has order item with id %s", orderItemId, orderItem.getId());
        checkAllServicesAndPaymentsAreCancelled(ctx);
    }

    protected void checkAllServicesAndPaymentsAreCancelled(StateContext<EHotelOrderState, HotelOrder> ctx) {
        long remainingServices = ctx.getWorkflowEntity().getOrderItems().stream()
                .filter(i -> ((HotelOrderItem) i).getPublicState() != HotelOrderItem.HotelOrderItemState.CANCELLED)
                .count();
        long remainingPayments = ctx.getWorkflowEntity().getPayments().stream()
                .filter(p -> p.getState() != EPaymentState.PS_CANCELLED)
                .count();
        if (remainingServices == 0 && remainingPayments == 0) {
            log.info("All services and payments are cancelled, cancelling the order");
            promoCodeApplicationService.freePromoCodeActivations(ctx.getWorkflowEntity());
            ctx.setState(EHotelOrderState.OS_CANCELLED);
        } else {
            log.info("Cancellation is not over yet: {} services and {} payments remain", remainingServices,
                    remainingPayments);
        }
    }
}
