package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.text.MessageFormat;
import java.time.Instant;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.plus.YandexPlusPromoService;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.orders.services.promo.mir2020.Mir2020PromoService;
import ru.yandex.travel.orders.services.promo.taxi2020.Taxi2020PromoService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.proto.TServiceInManualConfirmation;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TRenderDocuments;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.orders.workflow.payments.proto.TConfirmPayment;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.MessagingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.WorkflowMaintenanceService;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@RequiredArgsConstructor
@Slf4j
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TClearingInProcess.class,
        TInvoiceCleared.class,
})
public class WaitingConfirmationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState,
        HotelOrder> {
    private static final ImmutableSet<HotelOrderItem.HotelOrderItemState> POSSIBLE_STATES = ImmutableSet.of(
            HotelOrderItem.HotelOrderItemState.RESERVED, HotelOrderItem.HotelOrderItemState.CANCELLED,
            HotelOrderItem.HotelOrderItemState.CONFIRMED
    );
    private final StarTrekService starTrekService;

    private final WorkflowMaintenanceService workflowMaintenanceService;

    private final FinancialEventService financialEventService;

    private final Taxi2020PromoService taxi2020PromoService;

    private final Mir2020PromoService mir2020PromoService;

    private final UserOrderCounterService userOrderCounterService;

    private final YandexPlusPromoService yandexPlusPromoService;

    @HandleEvent
    public void handleConfirmed(TServiceConfirmed event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelOrder order = context.getWorkflowEntity();
        handleServiceStateChanged(context);
        HotelOrderItem orderItem = OrderCompatibilityUtils.getOnlyHotelOrderItem(order);
        financialEventService.registerConfirmedService(orderItem);
        taxi2020PromoService.registerConfirmedOrder(order);
        mir2020PromoService.registerConfirmedOrder(order);
        yandexPlusPromoService.registerConfirmedService(orderItem);
        userOrderCounterService.onOrderConfirmed(order.getId(), order.getDisplayType());
    }

    @HandleEvent
    public void handleCancelled(TServiceCancelled event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelOrder order = context.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1,
                "Exactly 1 order item is expected but got: %s", order.getOrderItems().size());
        handleServiceStateChanged(context);
    }

    private void handleServiceStateChanged(StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();
        if (!order.allHotelItemsAreIn(POSSIBLE_STATES)) {
            throw new RuntimeException(
                    MessageFormat.format("Order {0} is waiting for confirmation, but order item is not in allowed " +
                                    "state (got {1})", order.getId(),
                            order.getOrderItems().get(0).getItemState()) // working only with the case of one order item
            );
        }
        if (order.allHotelItemsAreIn(HotelOrderItem.HotelOrderItemState.CONFIRMED)) {
            ctx.setState(EHotelOrderState.OS_CONFIRMED);
            if (order.getActivePaymentWorkflowId() != null) {
                ctx.scheduleExternalEvent(order.getActivePaymentWorkflowId(), TConfirmPayment.newBuilder().build());
            }
            ctx.scheduleEvent(TRenderDocuments.newBuilder().build());
        } else if (order.atLeastOneHotelItemIsIn(HotelOrderItem.HotelOrderItemState.CANCELLED)) {
            Preconditions.checkState(order.getOrderItems().size() == 1, "Can't handle cancelled order item for order " +
                    "with more than 1 order item");
            ctx.setState(EHotelOrderState.OS_WAITING_CANCELLATION);
            if (order.getPayments().size() > 0) {
                order.getPayments().forEach(p ->
                        ctx.scheduleExternalEvent(p.getWorkflow().getId(),
                                TCancelPayment.newBuilder().setRefundIfAlreadyPaid(true).build()));
            } else {
                // schedule refund for all the invoices
                List<TrustInvoice> paidInvoices = order.getInvoices().stream()
                        .map(i -> (TrustInvoice) i)
                        .filter(i -> !TrustInvoice.UNPAID_INVOICE_STATES.contains(i.getInvoiceState()))
                        .collect(Collectors.toList());
                Preconditions.checkState(paidInvoices.size() == 1, "Exactly 1 paid invoice is expected");
                paidInvoices.forEach(i ->
                        ctx.scheduleExternalEvent(i.getWorkflow().getId(),
                                InvoiceUtils.buildFullRefund(i, "Cancelling the order")));
            }
        }
    }

    @HandleEvent
    public void handleServiceManualProcessing(TServiceInManualConfirmation event, StateContext<EHotelOrderState,
            HotelOrder> ctx) {
        log.warn("Order item is in MANUAL_CONFIRMATION, will pause the order and move it to MANUAL_PROCESSING");
        HotelOrder order = ctx.getWorkflowEntity();
        order.setState(EHotelOrderState.OS_MANUAL_PROCESSING);
        workflowMaintenanceService.pauseSupervisedRunningWorkflows(ctx.getWorkflowId());
    }

    @HandleEvent
    public void handleWorkflowCrashed(TWorkflowCrashed event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        if (WellKnownWorkflowEntityType.EXPEDIA_ORDER_ITEM.equalsValue(event.getEntityType())) {
            HotelOrder order = ctx.getWorkflowEntity();
            order.setState(EHotelOrderState.OS_MANUAL_PROCESSING);
            // we're not using try-catch construct here for the moment, as in case of crash it'll be handled by order
            // supervisor
            workflowMaintenanceService.pauseSupervisedRunningWorkflows(ctx.getWorkflowId());
            starTrekService.createIssueForOrderPaidNotConfirmed(order.getId(), event.getEventId(), Instant.now(), ctx);
        } else {
            MessagingUtils.throwOnUnmatchedEvent(event, ctx);
        }
    }
}
