package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.math.BigDecimal;
import java.text.MessageFormat;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.TrustRefundItem;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCancelled;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;


@RequiredArgsConstructor
@Slf4j
@IgnoreEvents(types = {
        TClearingInProcess.class,
        TInvoiceCleared.class,
        TInvoicePaymentStarted.class,
})
public class WaitingExtraPaymentStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState,
        HotelOrder> {
    private final AccountService accountService;
    private final FinancialEventService financialEventService;

    @HandleEvent
    public void handleVoucherCreated(TVoucherCreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelHandlerCommon.handleDocumentCreated(event.getVoucherType(), event.getVoucherUrl(), context);
    }

    @HandleEvent
    public void handleVoucherRecreated(TVoucherRecreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelHandlerCommon.handleDocumentCreated(event.getVoucherType(), event.getVoucherUrl(), context);
    }

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();

        Money totalCost = order.calculateTotalCost();
        var refundSum = order.getInvoices()
                .stream()
                .filter((invoice) -> invoice.getInvoiceState() != ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED)
                .map(invoice -> invoice
                        .getTrustRefunds()
                        .stream()
                        .map(trustRefund -> trustRefund
                                .getRefundItems()
                                .stream()
                                .map(TrustRefundItem::getRefundDelta)
                                .reduce(BigDecimal::add)
                                .orElse(BigDecimal.ZERO))
                        .reduce(BigDecimal::add)
                        .orElse(BigDecimal.ZERO))
                .reduce(BigDecimal::add)
                .orElse(BigDecimal.ZERO);
        var actualTotalCost = totalCost.subtract(Money.of(refundSum, order.getCurrency()));
        Money acquired = order.isFullyPostPaid() ? actualTotalCost : accountService.getAccountBalance(order.getAccount().getId());

        if (actualTotalCost.isEqualTo(acquired)) {
            Preconditions.checkState(order.getOrderItems().size() == 1,
                    "Exactly 1 order item is expected but got: %s", order.getOrderItems().size());
            HotelOrderItem orderItem = (HotelOrderItem) order.getOrderItems().get(0);
            Preconditions.checkState(orderItem.getPublicState() == HotelOrderItem.HotelOrderItemState.CONFIRMED,
                    "Service is not confirmed after extra payment");
            Payment payment = order.getPayments().stream()
                    .filter(pi -> pi.getId().toString().equals(event.getPaymentId()))
                    .findFirst().orElse(null);
            if (payment == null && !order.isFullyPostPaid())
                throw new IllegalStateException("No invoice found");
            financialEventService.registerExtraPaymentForService(orderItem, payment);
            for (OrderItem item : order.getOrderItems()) {
                ctx.scheduleExternalEvent(item.getWorkflow().getId(), event.toBuilder().build());
            }
            ctx.setState(EHotelOrderState.OS_CONFIRMED);
        } else {
            throw new RuntimeException(
                    MessageFormat.format("Money acquired for order {0} does not equal to the expected total (acquired" +
                            " {1}, expected {2})", order.getId(), acquired, totalCost)
            );
        }
    }

    @HandleEvent
    public void handlePaymentCancelled(TPaymentCancelled event,  StateContext<EHotelOrderState, HotelOrder> ctx) {
        log.info("Extra payment cancelled, returning order back to confirmed state");
        ctx.setState(EHotelOrderState.OS_CONFIRMED);
    }
}
