package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.text.MessageFormat;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@RequiredArgsConstructor
@Slf4j
@IgnoreEvents(types = {
        TClearingInProcess.class,
        TInvoiceCleared.class,
        TInvoicePaymentStarted.class,
})
public class WaitingPaymentStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState, HotelOrder> {

    private final AccountService accountService;

    private final Meters meters;

    private final PromoCodeApplicationService promoCodeApplicationService;

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();

        Money totalCost = order.calculateInitialPaymentAmount();
        Money acquired = accountService.getAccountBalance(order.getAccount().getId());

        if (totalCost.isEqualTo(acquired)) {
            if (!applyPromoCode(order)) {
                log.warn("Cannot apply promocode anymore. Cancelling the order. Order Id: {}", order.getId());
                ctx.scheduleEvent(TStartReservationCancellation.newBuilder()
                        .setReason(CancellationDetails.Reason.PROMO_ERROR.toString()).build());
            } else {
                ctx.setState(EHotelOrderState.OS_WAITING_CONFIRMATION);
                for (OrderItem item : order.getOrderItems()) {
                    ctx.scheduleExternalEvent(item.getWorkflow().getId(), TConfirmationStart.newBuilder().build());
                }
            }
        } else {
            throw new RuntimeException(
                    MessageFormat.format("Money acquired for order {0} does not equal to the expected total (acquired" +
                            " {1}, expected {2})", order.getId(), acquired, totalCost)
            );
        }
    }

    private boolean applyPromoCode(HotelOrder order) {
        if (order.getPromoCodeApplications()
                .stream()
                .anyMatch(application ->
                        application.getApplicationResultType() == EPromoCodeApplicationResultType.ART_SUCCESS)) {
            // not quite clear how to behave in case of multiple promo codes. But ATM this is not supported in the app
            promoCodeApplicationService.usePromoCodeActivationsAndApplyDiscounts(order, true);
            return order.getPromoCodeApplications()
                    .stream()
                    .allMatch(application ->
                            application.getApplicationResultType() == EPromoCodeApplicationResultType.ART_SUCCESS);
        }
        return true;
    }

    @HandleEvent
    public void handleStartReservationCancellation(TStartReservationCancellation event,
                                                   StateContext<EHotelOrderState, HotelOrder> ctx) {

        HotelOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        UUID orderItemWorkflowId = order.getOrderItems().get(0).getWorkflow().getId();
        ctx.setState(EHotelOrderState.OS_WAITING_CANCELLATION);
        if (order.getPayments().size() > 0) {
            log.info("Service is being cancelled, cancelling all the payments too");
            order.getPayments().forEach(p ->
                    ctx.scheduleExternalEvent(p.getWorkflow().getId(),
                            TCancelPayment.newBuilder().setRefundIfAlreadyPaid(true).build()));
        }
        ctx.scheduleExternalEvent(orderItemWorkflowId, TCancellationStart.newBuilder().setReason(event.getReason()).build());
    }

    @HandleEvent
    public void handleServiceCancelled(TServiceCancelled event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1, "Handling cancellation of multiple items is not " +
                "implemented");
        UUID orderItemId = ProtoChecks.checkStringIsUuid("order item id", event.getServiceId());
        OrderItem orderItem = order.getOrderItems().get(0);
        Preconditions.checkState(orderItem.getId().equals(orderItemId), "Got ServiceCancelled event for order item " +
                "%s, but order has order item with id %s", orderItemId, orderItem.getId());
        TrustInvoice currentInvoice = (TrustInvoice) order.getCurrentInvoice();
        if (order.getPayments().size() > 0) {
            log.info("Service cancelled prematurely, cancelling all the payments");
            order.getPayments().forEach(p ->
                    ctx.scheduleExternalEvent(p.getWorkflow().getId(),
                            TCancelPayment.newBuilder().setRefundIfAlreadyPaid(true).build()));
            ctx.setState(EHotelOrderState.OS_WAITING_CANCELLATION);
        } else {
            // legacy case - manually cancelling invoices
            if (currentInvoice != null) {
                if (currentInvoice.getState() == ETrustInvoiceState.IS_NEW || currentInvoice.getState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT) {
                    // waiting for refund after cancellation
                    ctx.setState(EHotelOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION);
                } else if (currentInvoice.getState() == ETrustInvoiceState.IS_HOLD) {
                    ctx.scheduleExternalEvent(currentInvoice.getWorkflow().getId(),
                            InvoiceUtils.buildFullRefund(currentInvoice));
                    ctx.setState(EHotelOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION);
                } else {
                    ctx.setState(EHotelOrderState.OS_CANCELLED);
                }
            } else {
                ctx.setState(EHotelOrderState.OS_CANCELLED);
            }
        }
    }
}
