package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.text.MessageFormat;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.PromoServiceHelper;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class WaitingReservationStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState, HotelOrder> {

    private final PromoCodeApplicationService promoCodeApplicationService;
    private final PromoServiceHelper promoServiceHelper;
    private final AuthorizationService authorizationService;
    private final UserOrderCounterService userOrderCounterService;

    private static final ImmutableSet<HotelOrderItem.HotelOrderItemState> POSSIBLE_STATES =
            ImmutableSet.of(HotelOrderItem.HotelOrderItemState.NEW,
                    HotelOrderItem.HotelOrderItemState.RESERVED, HotelOrderItem.HotelOrderItemState.CANCELLED);

    @HandleEvent
    public void handleReserved(TServiceReserved event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        handleServiceStateChanged(ctx);
    }

    @HandleEvent
    public void handleCancelled(TServiceCancelled event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        handleServiceStateChanged(ctx);
    }

    private void handleServiceStateChanged(StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();
        if (!order.allHotelItemsAreIn(POSSIBLE_STATES)) {
            throw new RuntimeException(
                    MessageFormat.format("Order {0} is waiting for reservation, but order item is not in allowed " +
                                    "state (got {1})", order.getId(),
                            order.getOrderItems().get(0).getItemState()) // working only with the case of one order item
            );
        }
        if (order.allHotelItemsAreIn(HotelOrderItem.HotelOrderItemState.RESERVED)) {
            log.info("All order items are reserved");
            order.refreshExpiresAt();
            applyPrePromoDiscountForOrder(order);
            promoCodeApplicationService.usePromoCodeActivationsAndApplyDiscounts(order, false);
            try {
                AuthorizedUser authorizedUser = authorizationService.getOrderOwner(order.getId());
                promoServiceHelper.validateAppliedPromoCampaigns(order,
                                authorizedUser, userOrderCounterService.getUserExistingOrderTypes(authorizedUser.getPassportId()))
                        .join();
                if (order.isFullyPostPaid()) {
                    ctx.setState(EHotelOrderState.OS_WAITING_CONFIRMATION);
                    for (OrderItem item : order.getOrderItems()) {
                        ctx.scheduleExternalEvent(item.getWorkflow().getId(), TConfirmationStart.newBuilder().build());
                    }
                } else {
                    ctx.setState(EHotelOrderState.OS_WAITING_PAYMENT);
                }
            } catch (Exception e) {
                log.error("Error validating promo campaigns with promo service, cancelling the order", e);
                order.getOrderItems().forEach(oi -> {
                    ctx.scheduleExternalEvent(
                            oi.getWorkflow().getId(),
                            TCancellationStart.newBuilder().setReason(CancellationDetails.Reason.PROMO_ERROR.toString()).build());
                });
            }
        } else if (order.atLeastOneHotelItemIsIn(HotelOrderItem.HotelOrderItemState.CANCELLED)) {
            Preconditions.checkArgument(order.getOrderItems().size() == 1, "Proper handler of multi-item order " +
                    "cancellation is not implemented");
            log.info("All items are cancelled, so the order is cancelled as well");


            ctx.setState(EHotelOrderState.OS_CANCELLED);
        }
    }

    private void applyPrePromoDiscountForOrder(HotelOrder order) {
        for (var i : order.getOrderItems()) {
            HotelOrderItem item = (HotelOrderItem) i;
            if (item.getHotelItinerary().getDiscount() != null) {
                Money discountReminder = item.getHotelItinerary().getDiscount();
                for (var fi : item.getFiscalItems()) {
                    if (fi.getMoneyAmountWithoutPlus().isGreaterThan(discountReminder)) {
                        fi.applyDiscount(discountReminder);
                        discountReminder = Money.zero(discountReminder.getCurrency());
                        break;
                    } else {
                        Money currentAmount = fi.getMoneyAmountWithoutPlus();
                        fi.applyDiscount(currentAmount);
                        discountReminder = discountReminder.subtract(currentAmount);
                    }
                }
                if (!discountReminder.isZero()) {
                    throw new RuntimeException("Non zero discount reminder: " + discountReminder);
                }
            }
        }
    }
}
