package ru.yandex.travel.orders.workflows.order.hotel.handlers;

import java.util.Map;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCanNotBeCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TStartInvoiceRefund;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@RequiredArgsConstructor
@Slf4j
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TClearingInProcess.class,
        TInvoiceCleared.class,
        TVoucherCreated.class,
        TVoucherRecreated.class,
        TPaymentCancelled.class,
        TPaymentCanNotBeCancelled.class,
})
public class WaitingServiceRefundStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EHotelOrderState, HotelOrder> {
    private final FinancialEventService financialEventService;

    private final PromoCodeApplicationService promoCodeApplicationService;

    private final UserOrderCounterService userOrderCounterService;

    @HandleEvent
    public void handleVoucherCreated(TVoucherCreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelHandlerCommon.handleDocumentCreated(event.getVoucherType(), event.getVoucherUrl(), context);
    }

    @HandleEvent
    public void handleVoucherRecreated(TVoucherRecreated event, StateContext<EHotelOrderState, HotelOrder> context) {
        HotelHandlerCommon.handleDocumentCreated(event.getVoucherType(), event.getVoucherUrl(), context);
    }

    @HandleEvent
    public void handleServiceRefundSuccess(TServiceRefunded event, StateContext<EHotelOrderState, HotelOrder> ctx) {
        HotelOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getPendingMoneyRefund() == null, "Other Money Refund is in progress");
        if (order.getPaymentSchedule() != null) {
            if (order.getPaymentSchedule().getState() != EPaymentState.PS_CANCELLED &&
                    order.getPaymentSchedule().getState() != EPaymentState.PS_FULLY_PAID &&
                    order.getPaymentSchedule().getState() != EPaymentState.PS_CANCELLATION_IN_PROGRESS &&
                    order.getPaymentSchedule().getState() != EPaymentState.PS_CANCELLATION_RETRY_POSTPONED) {
                throw new RetryableException("Non-cancelled payment of refunded service");
            }
        }
        ctx.setState(EHotelOrderState.OS_WAITING_INVOICE_REFUND);
        ctx.scheduleEvent(TStartInvoiceRefund.newBuilder()
                .putAllTargetFiscalItems(event.getTargetFiscalItemsMap())
                .putAllTargetFiscalItemsMarkup(event.getTargetFiscalItemsMarkupMap())
                .build());
        // TODO (mbobrov): move money refund creation to waiting invoice refund state
        String reason = event.getRefundDescription().length() != 0 ? event.getRefundDescription() : "Cancelling the order";
        MoneyRefund refund = MoneyRefund.createPendingRefundFromProto(ctx.getWorkflowEntity(),
                event.getTargetFiscalItemsMap(),
                event.getTargetFiscalItemsMarkupMap(),
                null, reason);
        refund.setState(MoneyRefundState.IN_PROGRESS);

        if (isFullRefund(event.getTargetFiscalItemsMap())) {
            promoCodeApplicationService.freePromoCodeActivations(order);
            userOrderCounterService.onOrderRefunded(order);
        }

        Preconditions.checkState(order.getOrderItems().size() == 1,
                "Exactly 1 order item is expected but got %s", order.getOrderItems().size());
        OrderItem orderItem = order.getOrderItems().get(0);
        if (!event.getSkipFinEvents()) {
            financialEventService.registerRefundedService(orderItem, event.getMoneyRefundMode());
        } else {
            log.warn("Service was refunded, but generation of financial events is skipped!");
        }
    }

    @HandleEvent
    public void handleNotificationComplete(TNotificationComplete event,
                                           StateContext<EHotelOrderState, HotelOrder> context) {
        log.info("Confirmation notification sent for order {}", context.getWorkflowEntity().getId());
    }

    private boolean isFullRefund(Map<Long, TPrice> targetFiscalItems) {
        for (var v : targetFiscalItems.entrySet()) {
            if (!ProtoUtils.fromTPrice(v.getValue()).isZero()) {
                return false;
            }
        }
        return true;
    }
}
