package ru.yandex.travel.orders.workflows.order.train;

import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.repository.MoneyRefundRepository;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.services.train.RebookingService;
import ru.yandex.travel.orders.services.train.TrainRefundLogService;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflows.invoice.trust.jobs.ClearingService;
import ru.yandex.travel.orders.workflows.order.train.handlers.CancelledStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.ConfirmedStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.ManualProcessingStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.NewStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.RefundedStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.WaitingCancellationStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.WaitingConfirmationStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.WaitingPaymentStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.WaitingRefundAfterCancellationStateHandler;
import ru.yandex.travel.orders.workflows.order.train.handlers.WaitingReservationStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.workflow.StatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.base.ProxyStateMapWorkflowEventHandler;
import ru.yandex.travel.workflow.repository.EntityStateTransitionRepository;

@Configuration
@RequiredArgsConstructor
public class TrainOrderWorkflowConfiguration {
    private final AccountService accountService;
    private final EntityStateTransitionRepository entityStateTransitionRepository;
    private final NotificationHelper notificationHelper;
    private final MoneyRefundRepository moneyRefundRepository;
    private final OrderRefundRepository orderRefundRepository;
    private final CheckMoneyRefundsService checkMoneyRefundsService;
    private final TrainTicketRefundRepository trainTicketRefundRepository;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final RebookingService rebookingService;
    private final TrainOrderItemRepository trainOrderItemRepository;
    private final TrainRefundLogService trainRefundLogService;
    private final TrainWorkflowService trainWorkflowService;
    private final FinancialEventService financialEventService;
    private final ClearingService clearingService;

    @Bean
    public WorkflowEventHandler<TrainOrder> trainOrderWorkflowEventHandler() {
        ImmutableMap.Builder<ETrainOrderState, StatefulWorkflowEventHandler<ETrainOrderState, TrainOrder>> builder =
                ImmutableMap.builder();
        builder
                .put(ETrainOrderState.OS_NEW, new NewStateHandler(trainWorkflowProperties))
                .put(ETrainOrderState.OS_WAITING_RESERVATION, new WaitingReservationStateHandler())
                .put(ETrainOrderState.OS_WAITING_PAYMENT, new WaitingPaymentStateHandler(accountService,
                        rebookingService, trainOrderItemRepository))
                .put(ETrainOrderState.OS_WAITING_CONFIRMATION, new WaitingConfirmationStateHandler(notificationHelper,
                        trainWorkflowService, financialEventService, clearingService))
                .put(ETrainOrderState.OS_WAITING_CANCELLATION, new WaitingCancellationStateHandler())
                .put(ETrainOrderState.OS_CANCELLED, new CancelledStateHandler())
                .put(ETrainOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION,
                        new WaitingRefundAfterCancellationStateHandler())
                .put(ETrainOrderState.OS_REFUNDED, new RefundedStateHandler(checkMoneyRefundsService))
                .put(ETrainOrderState.OS_CONFIRMED, new ConfirmedStateHandler(notificationHelper,
                        moneyRefundRepository, orderRefundRepository, checkMoneyRefundsService,
                        trainTicketRefundRepository, trainRefundLogService, financialEventService))
                .put(ETrainOrderState.OS_MANUAL_PROCESSING, new ManualProcessingStateHandler(moneyRefundRepository,
                        checkMoneyRefundsService));
        return new ProxyStateMapWorkflowEventHandler<>(builder.build(), entityStateTransitionRepository);
    }
}
