package ru.yandex.travel.orders.workflows.order.train;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.promo.DiscountApplicationConfig;
import ru.yandex.travel.orders.entities.promo.OrderGeneratedPromoCodes;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeHelpers;
import ru.yandex.travel.orders.repository.promo.PromoCodeRepository;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.TimeHelpers;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.promo.PromoCodeGenerationService;
import ru.yandex.travel.orders.services.promo.PromoCodeUnifier;
import ru.yandex.travel.orders.services.promo.simple.PromoSendMailData;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.WorkflowEntity;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Service
@RequiredArgsConstructor
@Slf4j
public class TrainWorkflowService {
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final PromoCodeGenerationService promoCodeGenerationService;
    private final PromoCodeRepository promoCodeRepository;
    private final SingleOperationService singleOperationService;
    private final Clock clock;

    @TransactionMandatory
    public <T extends Order & WorkflowEntity<?>> void scheduleTrainHotelPromoMail(T trainOrder) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(trainOrder),
                "Expected train order but got %s", trainOrder.getClass().getSimpleName());
        List<TrainOrderItem> trainOrderItems = OrderCompatibilityUtils.getTrainOrderItems(trainOrder);
        if (!trainWorkflowProperties.getPromoHotel2020().isEnabled() ||
                trainOrderItems.stream().allMatch(x -> x.getPayload().isSuburban())) {
            return;
        }
        var stationToSettlementGeoId = trainOrderItems
                .get(trainOrderItems.size() - 1)
                .getPayload()
                .getUiData()
                .getStationToSettlementGeoId();
        var dataBuilder = PromoSendMailData.builder()
                .parentEntityId(trainOrder.getId())
                .parentEntityType(trainOrder.getEntityType())
                .promoCampaignName("TrainsHotel2020")
                .emailCampaign(trainWorkflowProperties.getPromoHotel2020().getCampaign())
                .email(trainOrder.getEmail())
                .stationToSettlementGeoId(stationToSettlementGeoId);
        PromoCode promoCode = null;
        if (trainWorkflowProperties.isGeneratePersonalPromoCodes() &&
                trainOrder.getGeneratedPromoCodes() != null &&
                trainOrder.getGeneratedPromoCodes().getPromoCodes().size() > 0) {
            promoCode = trainOrder.getGeneratedPromoCodes().getPromoCodes().get(0);
        } else if (!Strings.isNullOrEmpty(trainWorkflowProperties.getCommonPromoCodeForHotels())){
            String code = PromoCodeUnifier.unifyCode(trainWorkflowProperties.getCommonPromoCodeForHotels());
            promoCode = promoCodeRepository.findByCodeEquals(code);
        } else {
            log.error("No promo codes for train promo mail found. Either generate one-time-promo-code or fill valid common-promo-code in config");
        }
        if (promoCode != null) {
            DiscountApplicationConfig discountApplicationConfig =
                    promoCode.getPromoAction().getDiscountApplicationConfig();
            LocalDateTime validTill = TimeHelpers.toLocalDateTime(PromoCodeHelpers.getPromoCodeValidTill(promoCode),
                    TimeHelpers.MSK_TZ);
            dataBuilder.promoCode(promoCode.getCode())
                    .codeAddsUpWithOtherActions(discountApplicationConfig.isAddsUpWithOtherActions())
                    .codeNominalType(promoCode.getNominalType().toString())
                    .codeNominal(promoCode.getNominal().intValueExact())
                    .codeMinTotalCost(discountApplicationConfig.getMinTotalCost().getNumber().intValueExact())
                    .codeValidTill(NotificationHelper.humanDate(validTill));
            singleOperationService.scheduleOperation(
                    "TrainsHotel2020Promo" + Instant.now(clock).toEpochMilli(),
                    OperationTypes.SIMPLE_PROMO_EMAIL_SENDER.getValue(),
                    dataBuilder.build(),
                    Instant.now().plus(trainWorkflowProperties.getPromoHotel2020().getDelay())
            );
        }
    }

    public <T extends Order & WorkflowEntity<?>> void createPromoCodeForHotels(T trainOrder) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(trainOrder),
                "Expected train order but got %s", trainOrder.getClass().getSimpleName());
        if (!trainWorkflowProperties.isGeneratePersonalPromoCodes()) {
            return;
        }
        if (Strings.isNullOrEmpty(trainWorkflowProperties.getPromoActionForHotelsName())) {
            log.error("Promo action for train-for-hotels promoCode not set. Please check config. Order: {}", trainOrder.getId());
            return;
        }

        String actionName = trainWorkflowProperties.getPromoActionForHotelsName();
        PromoCode promoCode = promoCodeGenerationService.generatePromoCodeForAction(actionName, LocalDate.now());
        if (promoCode != null) {
            if (trainOrder.getGeneratedPromoCodes() == null) {
                OrderGeneratedPromoCodes.createForOrder(trainOrder);
            }
            trainOrder.getGeneratedPromoCodes().addPromoCode(promoCode);
            log.info("Created promo code {} for order {} using action {}", promoCode.getCode(), trainOrder.getId(), actionName);
        } else {
            log.warn("Could not create promo code for order {} using action {}", trainOrder.getId(), actionName);
        }
    }
}
