package ru.yandex.travel.orders.workflows.order.train.handlers;

import java.time.Instant;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderOfficeRefund;
import ru.yandex.travel.orders.entities.TrainOrderUserRefund;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.promo.PromoCodeHelpers;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.repository.MoneyRefundRepository;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.services.train.TrainRefundLogService;
import ru.yandex.travel.orders.workflow.invoice.proto.TMoneyMarkup;
import ru.yandex.travel.orders.workflow.invoice.proto.TScheduleClearing;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.order.proto.EInvoiceRefundType;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.order.proto.TStartServiceRefund;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TChangeRegistrationStatus;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TUpdateTickets;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TRegistrationStatusChange;
import ru.yandex.travel.orders.workflow.train.proto.TRegistrationStatusChanged;
import ru.yandex.travel.orders.workflow.train.proto.TServiceOfficeRefunded;
import ru.yandex.travel.orders.workflow.train.proto.TServiceRefundFailed;
import ru.yandex.travel.orders.workflow.train.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.train.proto.TTrainTicketsUpdated;
import ru.yandex.travel.orders.workflow.train.proto.TUpdateTrainTickets;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TNotificationComplete.class})
@RequiredArgsConstructor
public class ConfirmedStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrainOrderState, TrainOrder> {
    private final NotificationHelper notificationHelper;
    private final MoneyRefundRepository moneyRefundRepository;
    private final OrderRefundRepository orderRefundRepository;
    private final CheckMoneyRefundsService checkMoneyRefundsService;
    private final TrainTicketRefundRepository trainTicketRefundRepository;
    private final TrainRefundLogService trainRefundLogService;
    private final FinancialEventService financialEventService;

    @HandleEvent
    public static void handleTicketsUpdated(TTrainTicketsUpdated event,
                                            StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        log.info("Service info refreshed for service {}", event.getServiceId());
    }

    @HandleEvent
    public void handle(TUpdateTrainTickets event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        order.getOrderItems().stream()
                .filter(orderItem -> orderItem.getPublicType() == EServiceType.PT_TRAIN)
                .forEach(orderItem -> ctx.scheduleExternalEvent(orderItem.getWorkflow().getId(),
                        TUpdateTickets.newBuilder().build()));
    }

    // service refund
    @HandleEvent
    public void handleStartRefund(TStartServiceRefund event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1, "Can't refund order with more than 1 order item");
        TrainOrderUserRefund refund = TrainOrderUserRefund.createForOrder(order);
        refund.setState(EOrderRefundState.RS_WAITING_SERVICE_REFUND);
        ctx.scheduleExternalEvent(order.getOrderItems().get(0).getWorkflow().getId(), TRefundStart.newBuilder()
                .setToken(event.getToken()).setOrderRefundId(refund.getId().toString()).build());
    }

    @HandleEvent
    public void handle(TServiceRefundFailed event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        OrderRefund orderRefund = orderRefundRepository.getOne(UUID.fromString(event.getOrderRefundId()));
        TrainOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        orderRefund.setState(EOrderRefundState.RS_FAILED);
    }

    @HandleEvent
    public void handle(ru.yandex.travel.orders.workflow.order.proto.TServiceRefundFailed event,
                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        OrderRefund orderRefund = orderRefundRepository.getOne(UUID.fromString(event.getOrderRefundId()));
        TrainOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        orderRefund.setState(EOrderRefundState.RS_FAILED);
    }

    @HandleEvent
    public void handle(TServiceRefunded event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        handleServiceRefunded(UUID.fromString(event.getOrderRefundId()), event.getTargetFiscalItemsMap(), null, ctx);
    }

    @HandleEvent
    public void handle(ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded event,
                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        handleServiceRefunded(UUID.fromString(event.getOrderRefundId()), event.getTargetFiscalItemsMap(),
                event.getTargetFiscalItemsMarkupMap(), ctx);
    }

    private void handleServiceRefunded(UUID orderRefundId, Map<Long, TPrice> fiscalItems,
                                       Map<Long, TMoneyMarkup> fiscalItemsMarkup,
                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        OrderRefund orderRefund = orderRefundRepository.findById(orderRefundId)
                .orElseThrow(() -> new NoSuchElementException("No such refund: " + orderRefundId));
        orderRefund.setState(EOrderRefundState.RS_WAITING_INVOICE_REFUND);
        MoneyRefund moneyRefund = MoneyRefund.createPendingRefundFromProto(order, fiscalItems,
                fiscalItemsMarkup, orderRefundId, "Refund");
        moneyRefundRepository.save(moneyRefund);
        boolean hasActiveInvoiceRefunds = order.getMoneyRefunds().stream()
                .anyMatch(r -> r.getState() == MoneyRefundState.IN_PROGRESS || r.getState() == MoneyRefundState.WAITING_CLEARING);
        if (!hasActiveInvoiceRefunds) {
            startInvoiceRefund(moneyRefund, ctx);
        }
        registerRefundEvents(orderRefund);
    }

    @HandleEvent
    public void handle(TServiceOfficeRefunded event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        var ticketRefundId = ProtoUtils.fromStringOrNull(event.getTicketRefundId());
        if (ticketRefundId == null) {
            // empty office refund message
            return;
        }
        var orderRefund = TrainOrderOfficeRefund.createForOrder(order);
        orderRefund.setState(EOrderRefundState.RS_WAITING_INVOICE_REFUND);
        orderRefund = orderRefundRepository.saveAndFlush(orderRefund);
        var ticketRefund = trainTicketRefundRepository.getOne(ticketRefundId);
        ticketRefund.setOrderRefund(orderRefund);
        MoneyRefund moneyRefund = MoneyRefund.createPendingRefundFromProto(order, event.getTargetFiscalItemsMap(),
                event.getTargetFiscalItemsMarkupMap(), orderRefund.getId(), "Refund");
        moneyRefundRepository.save(moneyRefund);
        boolean hasActiveInvoiceRefunds = order.getMoneyRefunds().stream()
                .anyMatch(r -> r.getState() == MoneyRefundState.IN_PROGRESS || r.getState() == MoneyRefundState.WAITING_CLEARING);
        if (!hasActiveInvoiceRefunds) {
            startInvoiceRefund(moneyRefund, ctx);
        }
        registerRefundEvents(orderRefund);
    }

    @HandleEvent
    public void handle(TClearingInProcess event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();

        Optional<MoneyRefund> pendingRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.IN_PROGRESS).findFirst();
        Preconditions.checkState(pendingRefund.isPresent(), "Pending refund must be present");
        pendingRefund.get().setState(MoneyRefundState.WAITING_CLEARING);
    }

    @HandleEvent
    public void handle(TInvoiceCleared event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();

        Optional<MoneyRefund> pendingRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.WAITING_CLEARING).findFirst();

        pendingRefund.ifPresent(r -> {
            r.setState(MoneyRefundState.PENDING);
            startInvoiceRefund(r, ctx);
        });
    }

    @HandleEvent
    public void handle(TInvoiceRefunded event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        MoneyRefund moneyRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.IN_PROGRESS)
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("No active MoneyRefund object"));
        moneyRefund.setState(MoneyRefundState.REFUNDED);

        OrderRefund orderRefund = order.getOrderRefunds().stream()
                .filter(x -> x.getId().equals(UUID.fromString(event.getOrderRefundId()))).findAny()
                .orElseThrow(() -> new NoSuchElementException("No such refund: " + event.getOrderRefundId()));
        orderRefund.setInvoiceRefundType(event.getRefundType());
        orderRefund.setState(EOrderRefundState.RS_REFUNDED);

        switch (orderRefund.getRefundType()) {
            case RT_TRAIN_OFFICE_REFUND:
                trainRefundLogService.logRefund(order, ((TrainOrderOfficeRefund) orderRefund).getTrainTicketRefunds());
                break;
            case RT_TRAIN_USER_REFUND:
                trainRefundLogService.logRefund(order, ((TrainOrderUserRefund) orderRefund).getTrainTicketRefunds());
                break;
        }

        if (orderRefund.getRefundType() == EOrderRefundType.RT_TRAIN_USER_REFUND) {
            //noinspection ConstantConditions
            var emailWorkflowId = notificationHelper.createWorkflowForTrainRefundEmail(order,
                    orderRefund, ((TrainOrderUserRefund) orderRefund).getTicketRefund());
            var sendEvent = TSend.newBuilder().build();
            ctx.scheduleExternalEvent(emailWorkflowId, sendEvent);
        } else if (orderRefund.getRefundType() == EOrderRefundType.RT_TRAIN_OFFICE_REFUND) {
            var emailWorkflowId = notificationHelper.createWorkflowForTrainRefundEmail(order,
                    orderRefund, ((TrainOrderOfficeRefund) orderRefund).getTicketRefund());
            var sendEvent = TSend.newBuilder().build();
            ctx.scheduleExternalEvent(emailWorkflowId, sendEvent);
        } else if (orderRefund.getRefundType() == EOrderRefundType.RT_TRAIN_INSURANCE_AUTO_RETURN) {
//            var emailWorkflowId = notificationHelper.createWorkflowForSuccessfulTrainEmail(order, orderRefund);
            var sendEvent = TSend.newBuilder().build();
//            ctx.scheduleExternalEvent(emailWorkflowId, sendEvent);
            try {
                var smsWorkflowId = notificationHelper.createWorkflowForSuccessfulTrainSms(order);
                ctx.scheduleExternalEvent(smsWorkflowId, sendEvent);
            } catch (Exception e) {
                log.error("Unable to create workflow for successful train SMS. SMS won't be sent", e);
            }
        }

        TrustInvoice currentInvoice = OrderUtils.getRequiredCurrentInvoice(ctx.getWorkflowEntity());
        Preconditions.checkArgument(currentInvoice.getId().equals(UUID.fromString(event.getInvoiceId())),
                "Current invoice %s, got invoice id %s in event", currentInvoice.getId(), event.getInvoiceId());

        // TODO (mbobrov, ganintsev): think of scheduling it further after refund
        ctx.scheduleExternalEvent(currentInvoice.getWorkflow().getId(),
                TScheduleClearing.newBuilder().setClearAt(ProtoUtils.fromInstant(Instant.now())).build());

        MoneyRefund pendingRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.PENDING)
                .findFirst()
                .orElse(null);
        if (pendingRefund != null) {
            startInvoiceRefund(pendingRefund, ctx);
        } else {
            if (order.getOrderItems().stream().allMatch(
                    oi -> oi.getItemState() == EOrderItemState.IS_CANCELLED)) {
                ctx.setState(ETrainOrderState.OS_CANCELLED);
            } else if (order.getOrderItems().stream().allMatch(
                    oi -> oi.getItemState() == EOrderItemState.IS_REFUNDED)) {
                ctx.setState(ETrainOrderState.OS_REFUNDED);
            }
        }

        PromoCodeHelpers.blacklistGeneratedPromoCodesForOrder(order);
    }

    @HandleEvent
    public void handleTrainManualRefund(TManualRefund event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder trainOrder = ctx.getWorkflowEntity();
        if (checkMoneyRefundsService.checkManualRefundAllowed(trainOrder)) {
            trainOrder.setState(ETrainOrderState.OS_MANUAL_PROCESSING);
            ctx.scheduleEvent(event);
        } else {
            log.warn("Money refund not allowed in state {} for order {}", trainOrder.getState(), trainOrder.getId());
            trainOrder.toggleUserActionScheduled(false);
        }
    }

    @HandleEvent
    public void handleRegistrationStatusChange(
            TRegistrationStatusChange event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();

        int itemsSize = order.getOrderItems().size();
        Preconditions.checkState(itemsSize == 1,
                "Exactly 1 order item is expected but got %s", itemsSize);

        var orderItem = order.getOrderItems().get(0);
        ctx.scheduleExternalEvent(orderItem.getWorkflow().getId(),
                TChangeRegistrationStatus.newBuilder()
                        .setEnabled(event.getEnabled())
                        .addAllBlankIds(event.getBlankIdsList()).build());
    }

    @HandleEvent
    public void handleRegistrationStatusChanged(
            TRegistrationStatusChanged event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
    }

    private void startInvoiceRefund(MoneyRefund refund, StateContext<ETrainOrderState, TrainOrder> ctx) {
        Preconditions.checkArgument(refund.getState() == MoneyRefundState.PENDING,
                "Expected PENDING MoneyRefund but got %s", refund.getState());
        refund.setState(MoneyRefundState.IN_PROGRESS);

        TrainOrder order = ctx.getWorkflowEntity();
        Invoice invoice = order.getCurrentInvoice();
        if (refund.getTargetFiscalItems().size() == 0) {
            log.info("Skip payment refund cause zero items");
            ctx.scheduleEvent(TInvoiceRefunded.newBuilder()
                    .setOrderRefundId(refund.getOrderRefundId().toString())
                    .setInvoiceId(invoice.getId().toString())
                    .setRefundType(EInvoiceRefundType.EIR_UNKNOWN)
                    .build());
        } else {
            Preconditions.checkState(order.getCurrentInvoice() != null, "Active invoice must be present in order");
            ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), RefundingUtils.createRefundEvent(refund));
        }
    }

    private void registerRefundEvents(OrderRefund orderRefund) {
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(orderRefund.getOrder());
        if (invoice.getProcessThroughYt() != null && invoice.getProcessThroughYt()) {
            TrainTicketRefund ticketRefund;
            if (orderRefund.getRefundType() == EOrderRefundType.RT_TRAIN_USER_REFUND) {
                ticketRefund = ((TrainOrderUserRefund) orderRefund).getTicketRefund();
            } else if (orderRefund.getRefundType() == EOrderRefundType.RT_TRAIN_OFFICE_REFUND) {
                ticketRefund = ((TrainOrderOfficeRefund) orderRefund).getTicketRefund();
            } else {
                throw new RuntimeException("Could not register refund events for refund type " + orderRefund.getRefundType());
            }
            if (ticketRefund != null) {
                financialEventService.registerPartialRefundForService(ticketRefund.getOrderItem(), orderRefund);
            }
        }
    }
}
