package ru.yandex.travel.orders.workflows.order.train.handlers;

import java.util.Optional;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.repository.MoneyRefundRepository;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class ManualProcessingStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrainOrderState, TrainOrder> {
    private final MoneyRefundRepository moneyRefundRepository;
    private final CheckMoneyRefundsService checkMoneyRefundsService;

    private static final Set<ETrustInvoiceState> INACTIVE_INVOICE_STATES = ImmutableSet.of(
            ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED,
            ETrustInvoiceState.IS_CANCELLED,
            ETrustInvoiceState.IS_CLEARED,
            ETrustInvoiceState.IS_REFUNDED
    );

    @HandleEvent
    public void handleTrainManualRefund(TManualRefund event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        var trainOrder = ctx.getWorkflowEntity();
        if (checkMoneyRefundsService.checkManualRefundAllowed(trainOrder)) {
            MoneyRefund moneyRefund = MoneyRefund.createPendingRefundFromProto(trainOrder,
                    event.getTargetFiscalItemsMap(), event.getTargetFiscalItemsMarkupMap(), null, event.getReason());
            moneyRefundRepository.save(moneyRefund);
            startInvoiceRefund(moneyRefund, ctx);
        } else {
            log.warn("Money refund not allowed in state {} for order {}", trainOrder.getState(), trainOrder.getId());
            trainOrder.setUserActionScheduled(false);
        }
    }

    @HandleEvent
    public void handleInvoiceRefunded(TInvoiceRefunded event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        Invoice invoice = order.getCurrentInvoice();
        MoneyRefund moneyRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.IN_PROGRESS)
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("No active MoneyRefund object"));
        moneyRefund.setState(MoneyRefundState.REFUNDED);

        Preconditions.checkState(invoice != null, "Current invoice must be present in order");

        var invoiceState = invoice.getInvoiceState();
        @SuppressWarnings("SuspiciousMethodCalls")
        boolean inactive = INACTIVE_INVOICE_STATES.contains(invoiceState);
        Preconditions.checkState(inactive, "Invoice %s has invalid state %s", invoice.getId(), invoiceState);
        order.setUserActionScheduled(false);

        order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.PENDING)
                .findFirst()
                .ifPresent(pendingRefund -> startInvoiceRefund(pendingRefund, ctx));
    }

    // duplicate

    @HandleEvent
    public void handle(TInvoiceCleared event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();

        Optional<MoneyRefund> pendingRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.WAITING_CLEARING).findFirst();

        pendingRefund.ifPresent(r -> {
            r.setState(MoneyRefundState.PENDING);
            startInvoiceRefund(r, ctx);
        });
    }

    @HandleEvent
    public void handleClearingInProcess(TClearingInProcess event, StateContext<ETrainOrderState, TrainOrder> ctx) {

        TrainOrder order = ctx.getWorkflowEntity();

        Optional<MoneyRefund> pendingRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.IN_PROGRESS).findFirst();
        Preconditions.checkState(pendingRefund.isPresent(), "Pending refund must be present");
        pendingRefund.get().setState(MoneyRefundState.WAITING_CLEARING);
    }

    private void startInvoiceRefund(MoneyRefund refund, StateContext<ETrainOrderState, TrainOrder> ctx) {
        Preconditions.checkArgument(refund.getState() == MoneyRefundState.PENDING,
                "Expected PENDING MoneyRefund but got %s", refund.getState());
        refund.setState(MoneyRefundState.IN_PROGRESS);

        TrainOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getCurrentInvoice() != null, "Active invoice must be present in order");
        Invoice invoice = order.getCurrentInvoice();

        ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), RefundingUtils.createRefundEvent(refund));
    }

}
