package ru.yandex.travel.orders.workflows.order.train.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TTrainTicketsUpdated;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TInvoiceCleared.class, TNotificationComplete.class})
@RequiredArgsConstructor
public class RefundedStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrainOrderState, TrainOrder> {
    private final CheckMoneyRefundsService checkMoneyRefundsService;

    @HandleEvent
    public void handleTrainManualRefund(TManualRefund event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        var trainOrder = ctx.getWorkflowEntity();
        if (checkMoneyRefundsService.checkManualRefundAllowed(trainOrder)) {
            trainOrder.setState(ETrainOrderState.OS_MANUAL_PROCESSING);
            ctx.scheduleEvent(event);
        } else {
            log.warn("Money refund not allowed in state {} for order {}", trainOrder.getState(), trainOrder.getId());
            trainOrder.toggleUserActionScheduled(false);
        }
    }

    @HandleEvent
    public void handleTicketsUpdated(TTrainTicketsUpdated event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        ConfirmedStateHandler.handleTicketsUpdated(event, ctx);
    }
}
