package ru.yandex.travel.orders.workflows.order.train.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TAddInsurance;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.orders.workflows.order.train.TrainWorkflowUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {
        TAddInsurance.class,
        TInvoicePaymentStarted.class,
        ru.yandex.travel.orders.workflow.train.proto.TStartReservationCancellation.class,
        ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation.class,
        TMoneyAcquireErrorOccurred.class
})
public class WaitingCancellationStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<ETrainOrderState, TrainOrder> {

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        order.setMoneyAcquired(true);
    }

    @HandleEvent
    public void handleServiceCancelled(ru.yandex.travel.orders.workflow.train.proto.TServiceCancelled event,
                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        handleServiceCancel(ctx);
    }

    @HandleEvent
    public void handleServiceCancelled(ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled event,
                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        handleServiceCancel(ctx);
    }

    static void handleServiceCancel(StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();

        Preconditions.checkState(order.getOrderItems().size() > 0,
                "Number of order items in train order %s is 0", order.getId());

        if (TrainWorkflowUtils.allItemsCancelled(order)) {
            TrustInvoice currentInvoice = (TrustInvoice) order.getCurrentInvoice();
            if (currentInvoice != null) {
                if (currentInvoice.getState() == ETrustInvoiceState.IS_NEW || currentInvoice.getState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT) {
                    // waiting for refund after cancellation
                    ctx.setState(ETrainOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION);
                } else if (order.isMoneyAcquired()) {
                    ctx.scheduleExternalEvent(currentInvoice.getWorkflow().getId(),
                            InvoiceUtils.buildFullRefund(currentInvoice));
                    ctx.setState(ETrainOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION);
                } else {
                    ctx.setState(ETrainOrderState.OS_CANCELLED);
                }
            } else {
                ctx.setState(ETrainOrderState.OS_CANCELLED);
            }
        }
    }
}
