package ru.yandex.travel.orders.workflows.order.train.handlers;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderInsuranceAutoRefund;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TServiceConfirmedWithoutInsurance;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.orders.workflows.invoice.trust.jobs.ClearingService;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.order.train.TrainWorkflowService;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils.buildFullRefundTargetFiscalItems;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsToProto;

@Slf4j
@IgnoreEvents(types = {
        ru.yandex.travel.orders.workflow.train.proto.TStartReservationCancellation.class,
        ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation.class
})
@RequiredArgsConstructor
public class WaitingConfirmationStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrainOrderState,
        TrainOrder> {
    private final NotificationHelper notificationHelper;
    private final TrainWorkflowService trainWorkflowService;
    private final FinancialEventService financialEventService;
    private final ClearingService clearingService;

    @HandleEvent
    public void handleServiceConfirmed(TServiceConfirmed event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        ctx.setState(ETrainOrderState.OS_CONFIRMED);
        TrainOrder trainOrder = ctx.getWorkflowEntity();

        trainWorkflowService.createPromoCodeForHotels(trainOrder);

        ensureOneOrderItem(trainOrder);
        TrainOrderItem trainOrderItem = (TrainOrderItem) trainOrder.getOrderItems().get(0);

        if (trainOrderItem.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
            TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(trainOrder);
            clearingService.clearInvoice(invoice);
        }

//        var emailWorkflowId = notificationHelper.createWorkflowForSuccessfulTrainEmail(trainOrder, null);
        var sendEvent = TSend.newBuilder().build();
//        ctx.scheduleExternalEvent(emailWorkflowId, sendEvent);
        try {
            var smsWorkflowId = notificationHelper.createWorkflowForSuccessfulTrainSms(trainOrder);
            ctx.scheduleExternalEvent(smsWorkflowId, sendEvent);
        } catch (Exception e) {
            log.error("Unable to create workflow for successful train SMS. SMS won't be sent", e);
        }
        trainWorkflowService.scheduleTrainHotelPromoMail(trainOrder);
        financialEventService.registerConfirmedService(trainOrderItem);
    }

    @HandleEvent
    public void handleServiceConfirmedWithoutInsurance(TServiceConfirmedWithoutInsurance event,
                                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        ensureOneOrderItem(order);

        OrderRefund refund = TrainOrderInsuranceAutoRefund.createForOrder(order);
        refund.setState(EOrderRefundState.RS_WAITING_INVOICE_REFUND);
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        ctx.scheduleEvent(TServiceRefunded.newBuilder()
                .setOrderRefundId(ProtoUtils.toStringOrEmpty(refund.getId()))
                .putAllTargetFiscalItems(convertTargetFiscalItemsToProto(buildFullRefundTargetFiscalItems(
                        invoice.getInvoiceItems().stream()
                                .filter(x -> x.getFiscalItemType() == FiscalItemType.TRAIN_INSURANCE)
                                .collect(toList()))))
                .build());
        ctx.setState(ETrainOrderState.OS_CONFIRMED);
        trainWorkflowService.scheduleTrainHotelPromoMail(order);
        financialEventService.registerConfirmedService(order.getOrderItems().get(0));
    }

    @HandleEvent
    public void handleServiceCancelled(ru.yandex.travel.orders.workflow.train.proto.TServiceCancelled event,
                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceCancelled(ctx);
    }

    @HandleEvent
    public void handleServiceCancelled(ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled event,
                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceCancelled(ctx);
    }

    private void serviceCancelled(StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();

        ensureOneOrderItem(order);

        // schedule full refund for all invoice items
        Invoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), InvoiceUtils.buildFullRefund(invoice));
        ctx.setState(ETrainOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION);
    }

    private void ensureOneOrderItem(TrainOrder order) {
        int orderItemsSize = order.getOrderItems().size();
        Preconditions.checkState(orderItemsSize == 1,
                "Number of items in train order %s is %s, expected 1",
                order.getId(), orderItemsSize);
    }
}
