package ru.yandex.travel.orders.workflows.order.train.handlers;

import java.text.MessageFormat;
import java.time.Instant;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.train.RebookingService;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TReservationStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsuranceReservationStart;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TAddInsurance;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = TInvoicePaymentStarted.class)
public class WaitingPaymentStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrainOrderState, TrainOrder> {

    private final AccountService accountService;
    private final RebookingService rebookingService;
    private final TrainOrderItemRepository trainOrderItemRepository;

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();

        Preconditions.checkState(order.getOrderItems().size() == 1, "More than one order item for order present");

        order.setMoneyAcquired(true);
        Money totalCost = order.calculateTotalCost();
        Money acquired = this.accountService.getAccountBalance(order.getAccount().getId());
        if (!totalCost.isEqualTo(acquired)) {
            throw new RuntimeException(
                    MessageFormat.format("Money acquired for order {0} does not equal to the expected total (acquired" +
                                    " {1}, expected {2})",
                            order.getId(), acquired, totalCost)
            );
        }

        ctx.setState(ETrainOrderState.OS_WAITING_CONFIRMATION);

        UUID orderItemWorkflowId = order.getOrderItems().get(0).getWorkflow().getId();
        ctx.scheduleExternalEvent(orderItemWorkflowId, TConfirmationStart.newBuilder().build());
    }

    @HandleEvent
    public void handleMoneyAcquireErrorOccured(TMoneyAcquireErrorOccurred event,
                                               StateContext<ETrainOrderState, TrainOrder> ctx) {
        // currently we ignore TMoneyAcquireErrorOccurred as we can retry payment
    }

    @HandleEvent
    public void handleAddInsurance(TAddInsurance event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(order.getOrderItems().size() == 1);
        order.setUserActionScheduled(false);
        ctx.setState(ETrainOrderState.OS_WAITING_RESERVATION);
        UUID orderItemWorkflowId = order.getOrderItems().get(0).getWorkflow().getId();
        ctx.scheduleExternalEvent(orderItemWorkflowId, TInsuranceReservationStart.newBuilder().build());
    }

    @HandleEvent
    public void handleServiceCancelled(ru.yandex.travel.orders.workflow.train.proto.TServiceCancelled event,
                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceCancelled(event.getServiceId(), ctx);
    }

    @HandleEvent
    public void handleServiceCancelled(ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled event,
                                       StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceCancelled(event.getServiceId(), ctx);
    }

    private void serviceCancelled(String serviceId, StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        if (order.isTrainRebookingEnabled() && order.getExpiresAt().isAfter(Instant.now())) {
            TrainOrderItem oldOrderItem = trainOrderItemRepository.getOne(UUID.fromString(serviceId));
            OrderItem rebookedItem = rebookingService.rebookItem(order, oldOrderItem);
            UUID orderItemWorkflowId = rebookedItem.getWorkflow().getId();
            ctx.scheduleExternalEvent(orderItemWorkflowId, TReservationStart.newBuilder().build());
            ctx.setState(ETrainOrderState.OS_WAITING_RESERVATION);
        } else {
            WaitingCancellationStateHandler.handleServiceCancel(ctx);
        }
    }

    @HandleEvent
    public void handleStartReservationCancellation(ru.yandex.travel.orders.workflow.train.proto.TStartReservationCancellation event,
                                                   StateContext<ETrainOrderState, TrainOrder> ctx) {
        startCancellation(ctx);
    }

    @HandleEvent
    public void handleStartReservationCancellation(ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation event,
                                                   StateContext<ETrainOrderState, TrainOrder> ctx) {
        startCancellation(ctx);
    }

    private void startCancellation(StateContext<ETrainOrderState, TrainOrder> ctx) {
        TrainOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        UUID orderItemWorkflowId = order.getOrderItems().get(0).getWorkflow().getId();
        ctx.setState(ETrainOrderState.OS_WAITING_CANCELLATION);
        ctx.scheduleExternalEvent(orderItemWorkflowId, TCancellationStart.newBuilder().build());
    }
}
