package ru.yandex.travel.orders.workflows.order.train.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TAddInsurance;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {TInvoicePaymentStarted.class, TMoneyAcquireErrorOccurred.class})
public class WaitingReservationStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<ETrainOrderState, TrainOrder> {

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        order.setMoneyAcquired(true);
    }

    @HandleEvent
    public void handleReserved(ru.yandex.travel.orders.workflow.train.proto.TServiceReserved event,
                               StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceReserved(ctx);
    }

    @HandleEvent
    public void handleReserved(ru.yandex.travel.orders.workflow.order.proto.TServiceReserved event,
                               StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceReserved(ctx);
    }

    private void serviceReserved(StateContext<ETrainOrderState, TrainOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        if (!order.isTrainRebookingEnabled()) {
            order.refreshExpiresAt();
        }
        ctx.setState(ETrainOrderState.OS_WAITING_PAYMENT);
        if (order.isMoneyAcquired()) {
            ctx.scheduleEvent(TMoneyAcquired.getDefaultInstance());
        }
    }

    @HandleEvent
    public void handleCancelled(ru.yandex.travel.orders.workflow.train.proto.TServiceCancelled event,
                                StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceCancelled(ctx);
    }

    @HandleEvent
    public void handleCancelled(ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled event,
                                StateContext<ETrainOrderState, TrainOrder> ctx) {
        serviceCancelled(ctx);
    }

    public void serviceCancelled(StateContext<ETrainOrderState, TrainOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        Invoice invoice = order.getCurrentInvoice();
        if (order.isMoneyAcquired()) {
            ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), InvoiceUtils.buildFullRefund(invoice));
            ctx.setState(ETrainOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION);
        } else {
            ctx.setState(ETrainOrderState.OS_CANCELLED);
        }
    }

    @HandleEvent
    public void handleAddInsurance(TAddInsurance event, StateContext<ETrainOrderState, TrainOrder> ctx) {
        // while rebooking ignore TAddInsurance. Sorry user, there is no insurance for you
    }
}
