package ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonApiClient;
import ru.yandex.avia.booking.services.tdapi.InvalidationReason;
import ru.yandex.travel.commons.http.apiclient.HttpApiException;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.services.avia.AviaApiProxy;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderAvailabilityCheckFailed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCheckAvilabilityFailed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCheckAvilabilityNaInvalidateTdCache;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotOrderItemCheckAvailabilityFailedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotItemState, AeroflotOrderItem> {
    private final AviaApiProxy aviaApiProxy;
    private final AviaTicketDaemonApiClient tdApiClient;
    private final AeroflotOrderItemHandlersHelper helper;

    @HandleEvent
    public void onCheckAvailabilityFailed(TAeroflotOrderItemCheckAvilabilityFailed event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        try {
            aviaApiProxy.refreshVariant(orderItem.getPayload().getVariantId());
            stateContext.scheduleEvent(TAeroflotOrderItemCheckAvilabilityNaInvalidateTdCache.newBuilder().build());
        } catch (HttpApiException e) {
            if (e.getStatusCode() == HttpStatus.NOT_FOUND.value()) {
                // 404 is an OK response, means we have no more available tariffs for the variant
                stateContext.scheduleEvent(TAeroflotOrderItemCheckAvilabilityNaInvalidateTdCache.newBuilder().build());
            } else if (e.getStatusCode() == HttpStatus.UNPROCESSABLE_ENTITY.value()) {
                // e.g. DepartureIsTooClose (external redirect is required), is ok too
                stateContext.scheduleEvent(TAeroflotOrderItemCheckAvilabilityNaInvalidateTdCache.newBuilder().build());
            } else {
                throw e;
            }
        }
    }

    @HandleEvent
    public void onCheckAvailabilityFailed(TAeroflotOrderItemCheckAvilabilityNaInvalidateTdCache event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        AeroflotServicePayload payload = orderItem.getPayload();
        try {
            tdApiClient.invalidateVariant(helper.buildInvalidateParams(payload), InvalidationReason.VARIANT_NOT_AVAILABLE);
        } catch (Exception e) {
            log.warn("Failed to call the variant invalidation api; orderId={}", orderItem.getOrder().getId(), e);
        }
        payload.setBookingFailureReason(BookingFailureReason.NOT_AVAILABLE);
        stateContext.setState(EAeroflotItemState.IS_CANCELLED);
        stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TAeroflotOrderAvailabilityCheckFailed.newBuilder().build());
    }
}
