package ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers;

import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonApiClient;
import ru.yandex.avia.booking.services.tdapi.InvalidationReason;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.proto.TReserveOrderItemReq;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderAvailabilityCheckCompleted;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCheckAvilabilityFailed;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotOfferPriceChangedException;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceProvider;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotOrderItemCheckAvailabilityStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotItemState, AeroflotOrderItem> {
    private final AeroflotServiceProvider provider;
    private final AviaTicketDaemonApiClient tdApiClient;
    private final AeroflotOrderItemHandlersHelper helper;

    @HandleEvent
    public void onCheckAvailabilityRequest(TReserveOrderItemReq event, StateContext<EAeroflotItemState,
            AeroflotOrderItem> stateContext) {
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        log.info("Reserving service; orderItem.id={}", orderItem.getId());
        try {
            if (provider.getAeroflotServiceForProfile(orderItem)
                    .checkAvailability(orderItem.getOrder().getId(), orderItem.getPayload(),
                            (TAviaTestContext) orderItem.getTestContext())) {
                stateContext.setState(EAeroflotItemState.IS_WAIT_TOKENIZATION);
                stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                        TAeroflotOrderAvailabilityCheckCompleted.newBuilder().build());
            } else {
                stateContext.setState(EAeroflotItemState.IS_CHECK_AVAILABILITY_FAILED);
                stateContext.scheduleEvent(TAeroflotOrderItemCheckAvilabilityFailed.newBuilder().build());
            }
        } catch (AeroflotOfferPriceChangedException e) {
            // in this case we expect the frontend to detect the changed price and ask the user for a confirmation
            AeroflotServicePayload payload = orderItem.getPayload();
            invalidateTdCacheSafe(orderItem.getOrder().getId(), payload);
            Preconditions.checkNotNull(e.getNewOffer(), "No new price info");
            payload.getVariant().setOffer(e.getNewOffer());
            orderItem.setPayload(payload);

            stateContext.setState(EAeroflotItemState.IS_WAIT_TOKENIZATION);
            stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                    TAeroflotOrderAvailabilityCheckCompleted.newBuilder().build());
        }
    }

    private void invalidateTdCacheSafe(UUID orderId, AeroflotServicePayload payload) {
        try {
            tdApiClient.invalidateVariant(helper.buildInvalidateParams(payload), InvalidationReason.PRICE_CHANGED);
        } catch (Exception e) {
            log.warn("Failed to call the variant invalidation api; orderId={}", orderId, e);
        }
    }
}
