package ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers;

import java.util.Map;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonApiClient;
import ru.yandex.avia.booking.services.tdapi.InvalidationReason;
import ru.yandex.travel.commons.http.apiclient.HttpApiException;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.services.avia.AviaApiProxy;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderCreationFailed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCreateOrderNaInvalidateTdCache;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCreateOrderNotAvailable;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCreateOrderPriceChanged;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

import static java.util.Map.entry;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotOrderItemCreateOrderFailedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotItemState, AeroflotOrderItem> {
    private final AviaApiProxy aviaApiProxy;
    private final AviaTicketDaemonApiClient tdApiClient;
    private final AeroflotOrderItemHandlersHelper helper;

    private final Map<BookingFailureReason, InvalidationReason> bookingFailureToInvalidationReasonMap = Map.ofEntries(
            entry(BookingFailureReason.PRICE_CHANGED, InvalidationReason.PRICE_CHANGED),
            entry(BookingFailureReason.NOT_AVAILABLE, InvalidationReason.VARIANT_NOT_AVAILABLE)
    );

    @HandleEvent
    public void onCreateOrderNotAvailable(TAeroflotOrderItemCreateOrderNotAvailable event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        try {
            aviaApiProxy.refreshVariant(orderItem.getPayload().getVariantId());
            stateContext.scheduleEvent(TAeroflotOrderItemCreateOrderNaInvalidateTdCache.newBuilder().build());
        } catch (HttpApiException e) {
            if (e.getStatusCode() == HttpStatus.NOT_FOUND.value()) {
                // 404 is an OK response, means we have no more available tariffs for the variant
                stateContext.scheduleEvent(TAeroflotOrderItemCreateOrderNaInvalidateTdCache.newBuilder().build());
            } else {
                throw e;
            }
        }
    }

    @HandleEvent
    public void onCreateOrderPriceChanged(TAeroflotOrderItemCreateOrderNaInvalidateTdCache event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        invalidateCacheAndCancel(stateContext, BookingFailureReason.NOT_AVAILABLE);
    }

    @HandleEvent
    public void onCreateOrderPriceChanged(TAeroflotOrderItemCreateOrderPriceChanged event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        invalidateCacheAndCancel(stateContext, BookingFailureReason.PRICE_CHANGED);
    }

    private void invalidateCacheAndCancel(StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext, BookingFailureReason failureReason) {
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        AeroflotServicePayload payload = orderItem.getPayload();
        try {
            tdApiClient.invalidateVariant(
                    helper.buildInvalidateParams(payload),
                    bookingFailureToInvalidationReasonMap.getOrDefault(failureReason, InvalidationReason.UNKNOWN)
            );
        } catch (Exception e) {
            log.warn("Failed to call the variant invalidation api; orderId={}", orderItem.getOrder().getId(), e);
        }
        payload.setBookingFailureReason(failureReason);
        stateContext.setState(EAeroflotItemState.IS_CANCELLED);
        stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TAeroflotOrderCreationFailed.newBuilder().build());
    }
}
