package ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers;

import java.time.Instant;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderExternalConfirmationOfCancelled;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderItemConfirmed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemPaymentConfirmationFailed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemPaymentConfirmed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemTicketsReleased;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@Slf4j
public class AeroflotOrderItemExternallyConfirmedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotItemState, AeroflotOrderItem> {
    @HandleEvent
    public void onEvent(TAeroflotOrderItemPaymentConfirmed event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        log.info("The order item has been confirmed after the external confirmation");
        stateContext.getWorkflowEntity().setConfirmedAt(Instant.now());
        stateContext.setState(EAeroflotItemState.IS_CONFIRMED);
        // not copying the pnr as external confirmation can only happen for orders with pnr
        stateContext.scheduleExternalEvent(stateContext.getWorkflowEntity().getOrderWorkflowId(),
                TAeroflotOrderItemConfirmed.newBuilder().build());
    }

    @HandleEvent
    public void onPaymentFailed(TAeroflotOrderItemPaymentConfirmationFailed event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        // can reach this state during the order expiration procedure
        AeroflotOrderItemHandlersHelper.setFailureReason(stateContext.getWorkflowEntity(), event);
        stateContext.getWorkflowEntity().setConfirmedAt(Instant.now());
        stateContext.setState(EAeroflotItemState.IS_CONFIRMED);
        stateContext.scheduleExternalEvent(stateContext.getWorkflowEntity().getOrderWorkflowId(),
                TAeroflotOrderExternalConfirmationOfCancelled.newBuilder().build());
    }

    @HandleEvent
    public void onTicketsReleased(TAeroflotOrderItemTicketsReleased event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        log.info("Assigning the externally confirmed tickets; tickets={}", event.getTicketsList());
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        AeroflotServicePayload payload = orderItem.getPayload();
        AeroflotWorkflowUtils.updateTickets(payload, event.getTicketsList());
        orderItem.setPayload(payload);
    }
}
