package ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.avia.booking.enums.PassengerCategory;
import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotGateway;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOriginDestination;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.SearchData;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.SearchData.SearchParams;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonApiClient;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonSegment;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemPaymentConfirmationFailed;

import static java.util.stream.Collectors.toList;

@Component
@Slf4j
public class AeroflotOrderItemHandlersHelper {
    public AviaTicketDaemonApiClient.InvalidateVariantParams buildInvalidateParams(AeroflotServicePayload payload) {
        SearchData sd = payload.getVariant().getSearchData();
        SearchParams sp = sd.getSearchParams();
        if (sp == null) {
            throw new RuntimeException("SearchParams weren't passed for the current order");
        }
        Map<PassengerCategory, Integer> categories = new HashMap<>();
        for (TravellerInfo traveller : payload.getTravellers()) {
            categories.put(traveller.getCategory(), categories.getOrDefault(traveller.getCategory(), 0) + 1);
        }
        List<List<AviaTicketDaemonSegment>> legs = new ArrayList<>();
        for (AeroflotOriginDestination od : payload.getVariant().getOriginDestinations()) {
            legs.add(payload.getVariant().getOriginDestinationSegments(od.getId()).stream()
                    .map(seg -> new AviaTicketDaemonSegment(
                            seg.getMarketingCarrier().getAirlineId(),
                            seg.getMarketingCarrier().getFlightNumber(),
                            seg.getDeparture().getDateTime()
                    ))
                    .collect(toList()));
        }
        Preconditions.checkArgument(legs.size() >= 1 && legs.size() <= 2,
                "Exactly 1 or 2 legs expected; legs=%s", legs);
        return AviaTicketDaemonApiClient.InvalidateVariantParams.builder()
                .nationalVersion(sp.getNationalVersion())
                .lang(sp.getLang())
                .klass(sp.getKlass())
                .pointFrom(sp.getPointFrom())
                .pointTo(sp.getPointTo())
                .adults(categories.getOrDefault(PassengerCategory.ADULT, 0))
                .children(categories.getOrDefault(PassengerCategory.CHILD, 0))
                .infants(categories.getOrDefault(PassengerCategory.INFANT, 0))
                .partner(AeroflotGateway.AEROFLOT_PARTNER_ID)
                .forward(legs.get(0))
                .backward(legs.size() > 1 ? legs.get(1) : null)
                .variantTag(sd.getVariantTag())
                .build();
    }

    public static void setFailureReason(AeroflotOrderItem orderItem,
                                        TAeroflotOrderItemPaymentConfirmationFailed event) {
        AeroflotServicePayload payload = orderItem.getPayload();
        BookingFailureReason failureReason = null;
        if (event.getFailed()) {
            failureReason = BookingFailureReason.PAYMENT_FAILED;
        } else if (event.getTimedOut()) {
            failureReason = BookingFailureReason.PAYMENT_TIMED_OUT;
        }
        if (failureReason != null) {
            if (payload.getBookingFailureReason() != null) {
                log.warn("The order item already has a failure reason, overriding it: {} -> {}",
                        payload.getBookingFailureReason(), failureReason);
            } else {
                log.info("Booking failure reason: {}", failureReason);
            }
            payload.setBookingFailureReason(failureReason);
            // making sure the entity is marked as dirty and will be saved
            orderItem.setPayload(payload);
            if (failureReason == BookingFailureReason.PAYMENT_TIMED_OUT) {
                orderItem.setExpired(true);
            }
        } else {
            log.warn("Unknown payment failure reason");
        }
    }
}
