package ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers;

import java.time.Instant;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderItemCancelled;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderItemConfirmed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemPaymentConfirmationFailed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemPaymentConfirmed;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemTicketsReleased;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@Slf4j
public class AeroflotOrderItemWaitConfirmationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotItemState, AeroflotOrderItem> {
    @HandleEvent
    public void onEvent(TAeroflotOrderItemPaymentConfirmed event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        log.info("The order item has been confirmed; order_item_id={}", stateContext.getWorkflowEntity().getId());
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        orderItem.setConfirmedAt(Instant.now());
        if (orderItem.getAviaPnr() == null) {
            String pnr = Strings.emptyToNull(event.getPnr());
            if (pnr != null) {
                AeroflotOrderRef orderRef = orderItem.getPayload().getBookingRef();
                orderRef.setPnr(pnr);
                orderItem.setAviaPnr(AeroflotWorkflowUtils.composeAviaPnr(orderRef.getPnr(), orderRef.getPnrDate()));
            } else {
                log.warn("Successfully paid order without PNR!");
            }
        }
        stateContext.setState(EAeroflotItemState.IS_CONFIRMED);
        stateContext.scheduleExternalEvent(stateContext.getWorkflowEntity().getOrderWorkflowId(),
                TAeroflotOrderItemConfirmed.newBuilder().build());
    }

    @HandleEvent
    public void onPaymentFailed(TAeroflotOrderItemPaymentConfirmationFailed event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        AeroflotOrderItemHandlersHelper.setFailureReason(stateContext.getWorkflowEntity(), event);
        stateContext.setState(EAeroflotItemState.IS_CANCELLED);
        stateContext.scheduleExternalEvent(stateContext.getWorkflowEntity().getOrderWorkflowId(),
                TAeroflotOrderItemCancelled.newBuilder().build());
    }

    @HandleEvent
    public void onTicketsReleased(TAeroflotOrderItemTicketsReleased event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        log.info("Assigning the confirmed tickets; tickets={}", event.getTicketsList());
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        AeroflotServicePayload payload = orderItem.getPayload();
        AeroflotWorkflowUtils.updateTickets(payload, event.getTicketsList());
        orderItem.setPayload(payload);
        stateContext.setState(EAeroflotItemState.IS_EXTERNALLY_CONFIRMED);
    }
}
