package ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers;

import java.time.Instant;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotPaymentException;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderCreateResult;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.availability.PriceChangedException;
import ru.yandex.avia.booking.partners.gateways.model.availability.VariantNotAvailableException;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureException;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.partners.gateways.model.payment.PaymentFailedException;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.services.DeduplicationService;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderCreatedAndPaid;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderCreationFailed;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderItemCancelled;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderPaymentConfirmation;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCardTokenized;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCreateOrderNotAvailable;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCreateOrderPriceChanged;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemTokenizationFailed;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowUtils;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceProvider;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotOrderItemWaitTokenizationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotItemState, AeroflotOrderItem> {
    private final AeroflotServiceProvider provider;
    private final DeduplicationService deduplicationService;

    @HandleEvent
    public void onEvent(TAeroflotOrderItemCardTokenized event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        log.info("Creating a new aeroflot order; order_item_id={}", orderItem.getId());
        deduplicationService.registerAtMostOnceCall(UUID.fromString(event.getDeduplicationKey()));

        Order order = orderItem.getOrder();
        AeroflotServicePayload payload = orderItem.getPayload();
        String offerId = payload.getVariant().getOffer().getId();
        try {
            Invoice invoice = AeroflotWorkflowUtils.getOnlyInvoice(order);
            String redirectUrl = invoice.getConfirmationReturnPath();

            AeroflotOrderCreateResult result = provider.getAeroflotServiceForProfile(orderItem)
                    .createOrderAndStartPayment(order.getId(), payload, redirectUrl, event.getTokenizedCard());
            setOrderRef(orderItem, result.getOrderRef());

            if (result.is3dsRequired()) {
                stateContext.setState(EAeroflotItemState.IS_WAIT_CONFIRMATION);
                stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                        TAeroflotOrderPaymentConfirmation.newBuilder().setConfirmationUrl(result.getConfirmationUrl()).build());
            } else if (result.isPaid() && Strings.isNullOrEmpty(result.getOrderRef().getPnr())) {
                log.info("The order is paid but no PNR is available. We'll be waiting until it's finally available");
                stateContext.setState(EAeroflotItemState.IS_WAIT_CONFIRMATION);
                stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                        TAeroflotOrderPaymentConfirmation.newBuilder().build());
            } else if (result.isPaid()) {
                Preconditions.checkNotNull(result.getOrderRef().getPnr(), "The order is paid but no PNR assigned");
                stateContext.getWorkflowEntity().setConfirmedAt(Instant.now());
                stateContext.setState(EAeroflotItemState.IS_CONFIRMED);
                stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TAeroflotOrderCreatedAndPaid.newBuilder().build());
            } else {
                throw new RuntimeException("Unexpected order status: " + result.getStatusCode());
            }
        } catch (VariantNotAvailableException e) {
            log.info("The variant isn't available anymore: offerId={}", offerId);
            stateContext.setState(EAeroflotItemState.IS_CREATE_ORDER_NOT_AVAILABLE);
            stateContext.scheduleEvent(TAeroflotOrderItemCreateOrderNotAvailable.newBuilder().build());
        } catch (PriceChangedException e) {
            log.info("Price for the variant has changed: offerId={}", offerId);
            stateContext.setState(EAeroflotItemState.IS_PRICE_CHANGED);
            stateContext.scheduleEvent(TAeroflotOrderItemCreateOrderPriceChanged.newBuilder().build());
        } catch (PaymentFailedException e) {
            log.info("Order payment has failed: offerId={}, reason={}", offerId, e.getReason());
            if (e instanceof AeroflotPaymentException) {
                AeroflotOrderRef orderRef = ((AeroflotPaymentException) e).getOrderRef();
                if (orderRef != null) {
                    setOrderRef(orderItem, orderRef);
                }
            }
            payload.setBookingFailureReason(BookingFailureReason.PAYMENT_FAILED);
            orderItem.setPayload(payload);
            stateContext.setState(EAeroflotItemState.IS_CANCELLED);
            stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TAeroflotOrderCreationFailed.newBuilder().build());
        } catch (BookingFailureException e) {
            log.info("Failed to create the order: offerId={}, reason={}", offerId, e.getReason());
            payload.setBookingFailureReason(e.getReason());
            orderItem.setPayload(payload);
            stateContext.setState(EAeroflotItemState.IS_CANCELLED);
            stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TAeroflotOrderCreationFailed.newBuilder().build());
        } catch (RetryableException e) {
            log.info("Removing the deduplication key because of a safe retryable exception: {}", e.getMessage());
            deduplicationService.unregisterAtMostOnceCall(UUID.fromString(event.getDeduplicationKey()));
            throw e;
        }
    }

    private void setOrderRef(AeroflotOrderItem orderItem, AeroflotOrderRef orderRef) {
        orderItem.getPayload().setBookingRef(orderRef);
        if (!Strings.isNullOrEmpty(orderRef.getPnr())) {
            orderItem.setAviaPnr(AeroflotWorkflowUtils.composeAviaPnr(orderRef.getPnr(), orderRef.getPnrDate()));
        }
    }

    @HandleEvent
    public void onEvent(TAeroflotOrderItemTokenizationFailed event, StateContext<EAeroflotItemState, AeroflotOrderItem> stateContext) {
        AeroflotOrderItem orderItem = stateContext.getWorkflowEntity();
        var payload = orderItem.getPayload();
        payload.setBookingFailureReason(BookingFailureReason.TOKENIZATION_FAILED);
        orderItem.setPayload(payload);

        stateContext.setState(EAeroflotItemState.IS_CANCELLED);
        stateContext.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                TAeroflotOrderItemCancelled.newBuilder().build());
    }
}
