package ru.yandex.travel.orders.workflows.orderitem.aeroflot.mq;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.AeroflotMqMessage;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.AeroflotOrderItemRepository;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqData;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqParser;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.mq.proto.EAeroflotMqMessageState;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.mq.proto.TAeroflotMqMessageProcessingStarted;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemTicketsReleased;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotMqMessageNewStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotMqMessageState,
        AeroflotMqMessage> {
    private final AeroflotMqParser parser;
    private final AeroflotOrderItemRepository repository;
    private final StarTrekService starTrekService;

    @HandleEvent
    public void onProcessingStarted(TAeroflotMqMessageProcessingStarted event, StateContext<EAeroflotMqMessageState,
            AeroflotMqMessage> ctx) {
        AeroflotMqMessage message = ctx.getWorkflowEntity();
        AeroflotMqData order = parser.parseMessage(message.getData());
        String aviaPnr = AeroflotWorkflowUtils.composeAviaPnr(order.getPnr(), order.getPnrDate());
        log.info("Processing order sync message from Aeroflot; avia_pnr={}, paid={}", aviaPnr, order.isPaid());

        AeroflotOrderItem item = repository.findFirstByAviaPnrOrderByCreatedAtDesc(aviaPnr);
        if (item != null) {
            if (order.isPaid()) {
                // created & paid
                Preconditions.checkArgument(!order.getPassengers().isEmpty(), "Passengers list is empty: pnr=%s",
                        aviaPnr);
                String ownerLastName = order.getPassengers().get(0).getLastName();
                ctx.scheduleExternalEvent(item.getWorkflow().getId(), TAeroflotOrderItemTicketsReleased.newBuilder()
                        .addAllTickets(order.getTickets())
                        .setOwnerLastName(ownerLastName)
                        .build());
            }
            // else => created but not paid yet
            message.setOrderItemId(item.getId());
        } else if (order.isPaid()) {
            // will track only paid orders as there cases when the OrderCreateRQ call has failed but the mq
            // notification still arrives
            log.warn("Lost paid order has been found: avia_pnr={}", aviaPnr);
            // TRAVELBACK-286: temporary solution for masked ticket data
            //UUID ticketId = starTrekService.createIssueForAeroflotLostOrder(order, ctx);
            //message.setTicketId(ticketId);
        } else {
            log.warn("Lost unpaid order has been found: avia_pnr={}", aviaPnr);
        }
    }
}
