package ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider;

import java.time.Clock;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.validation.annotation.Validated;

import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotGateway;
import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotProviderProperties;
import ru.yandex.avia.booking.partners.gateways.aeroflot.demo.AeroflotDemoCardTokenizer;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;
import ru.yandex.travel.orders.repository.mock.MockAeroflotOrderRepository;

@Configuration
@Slf4j
@RequiredArgsConstructor
@EnableConfigurationProperties(MockAeroflotProperties.class)
public class AeroflotProviderConfiguration {
    private final Environment environment;

    @Bean
    @ConfigurationProperties(value = "aeroflot", ignoreUnknownFields = false)
    @Validated
    public AeroflotProviderProperties aeroflotProviderProperties() {
        return new AeroflotProviderProperties();
    }

    @Bean
    public AeroflotGateway aeroflotGateway(
            AeroflotProviderProperties properties,
            AsyncHttpClientWrapper aeroflotAhcClientWrapper,
            Clock clock
    ) {
        return new AeroflotGateway(properties, aeroflotAhcClientWrapper, clock);
    }

    @Bean
    public AeroflotDemoCardTokenizer aeroflotDemoCardTokenizer() {
        return new AeroflotDemoCardTokenizer();
    }

    @Bean
    @ConditionalOnProperty("mock-aeroflot.enabled")
    public MockAeroflotServiceAdapter mockAeroflotServiceAdapter(MockAeroflotOrderRepository mockAeroflotOrderRepository,
                                                                 AeroflotProviderProperties aeroflotProviderProperties,
                                                                 TransactionTemplate transactionTemplate) {
        if (environment.acceptsProfiles("prod")) {
            throw new RuntimeException("\n\nATTENTION: MOCK AEROFLOT MUST BE DISABLED FOR PROD ENVIRONMENT\n\n");
        }
        return new MockAeroflotServiceAdapter(mockAeroflotOrderRepository, aeroflotProviderProperties,
                transactionTemplate);
    }

    @Bean
    public AeroflotServiceProvider aeroflotService(AeroflotGateway aeroflotGateway,
                                                   AeroflotProviderProperties aeroflotProviderProperties,
                                                   ObjectProvider<MockAeroflotServiceAdapter> mockAeroflotObjectProvider) {
        return orderItem -> {
            if (orderItem.getTestContext() instanceof TAviaTestContext && !environment.acceptsProfiles("prod")) {
                MockAeroflotServiceAdapter mockAdapter = mockAeroflotObjectProvider.getIfUnique();
                return Preconditions.checkNotNull(mockAdapter,
                        "Mock aeroflot service adapter requested but not present. Set mock-aeroflot.enabled: true");
            }
            return new AeroflotServiceAdapter(aeroflotGateway, aeroflotProviderProperties);
        };
    }
}
