package ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider;

import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.avia.booking.partners.gateways.BookingRetryableException;
import ru.yandex.avia.booking.partners.gateways.BookingTestingScenarios;
import ru.yandex.avia.booking.partners.gateways.BookingTooManyRequestsException;
import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotGateway;
import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotProviderProperties;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderCreateResult;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTotalOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.model.availability.VariantNotAvailableException;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.avia.booking.remote.RpcContext;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;
import ru.yandex.travel.workflow.exceptions.RetryableException;
import ru.yandex.travel.workflow.exceptions.TooManyRequestsRetryableException;

@RequiredArgsConstructor
@Slf4j
public class AeroflotServiceAdapter implements AeroflotService {
    private final AeroflotGateway gateway;
    private final AeroflotProviderProperties properties;

    public boolean checkAvailability(UUID orderItemId, AeroflotServicePayload payload, TAviaTestContext testContext) {
        Preconditions.checkArgument(testContext == null, "TestContext should be null or MockAeroflot should be present");
        String offerId = null;
        try {
            offerId = payload.getVariant().getOffer().getId();
            log.info("Checking availability for offer_id={}", offerId);
            if (properties.isEnableTestingScenarios()) {
                checkAvailabilityCheckTestingScenarios(payload);
            }
            AeroflotVariant token = payload.getVariant();
            AeroflotVariant updated = gateway.checkAvailabilitySingle(token);
            Money oldPrice = token.getOffer().getTotalPrice();
            Money newPrice = updated.getOffer().getTotalPrice();
            if (!oldPrice.equals(newPrice)) {
                String message = String.format("The offer price has changed; offer_id=%s, old=%s, new=%s",
                        offerId, oldPrice, newPrice);
                log.info(message);
                throw new AeroflotOfferPriceChangedException(message, updated.getOffer());
            }
            log.info("The variant is still available; offer_id={}", offerId);
            return true;
        } catch (VariantNotAvailableException e) {
            log.info("The variant is not available anymore; offer_id={}; e.msg={}", offerId, e.getMessage());
            return false;
        } catch (BookingRetryableException e) {
            throw new RetryableException(e);
        } catch (BookingTooManyRequestsException e) {
            throw new TooManyRequestsRetryableException(e);
        }
    }

    public AeroflotOrderCreateResult createOrderAndStartPayment(UUID yaOrderId, AeroflotServicePayload payload,
                                                                String redirectUrl, String tokenizedCard) {
        try {
            Preconditions.checkArgument(payload.getTravellers().size() > 0, "Not empty list of traveller ids is expected");
            if (properties.isEnableTestingScenarios()) {
                checkCreateOrderTestingScenarios(payload);
            }
            AeroflotVariant variant = payload.getVariant();
            RpcContext rpcContext = RpcContext.builder().orderId(yaOrderId).build();
            return gateway.initPayment(variant, payload.getTravellers(), tokenizedCard, payload.getClientInfo(), redirectUrl, rpcContext);
        } catch (BookingRetryableException e) {
            throw new RetryableException(e);
        }
    }

    public AeroflotOrderCreateResult getOrderStatus(UUID yaOrderId, AeroflotServicePayload payload) {
        try {
            return gateway.getOrderStatus(payload, RpcContext.empty());
        } catch (BookingRetryableException e) {
            throw new RetryableException(e);
        } catch (BookingTooManyRequestsException e) {
            throw new TooManyRequestsRetryableException(e);
        }
    }

    static void checkAvailabilityCheckTestingScenarios(AeroflotServicePayload payload) {
        TravellerInfo traveller = payload.getTravellers().get(0);
        String firstName = traveller.getFirstName();
        if (BookingTestingScenarios.BOOKING_NEW_PRICE.equalsIgnoreCase(firstName)) {
            AeroflotTotalOffer offer = payload.getVariant().getOffer();
            Money wrongPreliminaryPrice = offer.getTotalPrice().divide(2);
            offer.setTotalPrice(wrongPreliminaryPrice);
            payload.setPreliminaryCost(wrongPreliminaryPrice);
        }
    }

    static void checkCreateOrderTestingScenarios(AeroflotServicePayload payload) {
        TravellerInfo traveller = payload.getTravellers().get(0);
        String firstName = traveller.getFirstName();
        if (BookingTestingScenarios.BOOKING_NOT_FOUND.equalsIgnoreCase(firstName)) {
            throw new VariantNotAvailableException("Testing Scenario: not available anymore");
        }
        if (BookingTestingScenarios.BOOKING_NEW_UNHANDLED.equalsIgnoreCase(firstName)) {
            throw new RuntimeException("Testing Scenario: unhandled");
        }
    }
}
