package ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomStringUtils;
import org.javamoney.moneta.Money;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.avia.booking.partners.gateways.BookingRetryableException;
import ru.yandex.avia.booking.partners.gateways.BookingTooManyRequestsException;
import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotPaymentException;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderCreateResult;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderStatus;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotPriceDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTotalOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.model.availability.PriceChangedException;
import ru.yandex.avia.booking.partners.gateways.model.availability.VariantNotAvailableException;
import ru.yandex.avia.booking.partners.gateways.model.payment.PaymentFailureReason;
import ru.yandex.travel.orders.commons.proto.EAviaCheckAvailabilityOutcome;
import ru.yandex.travel.orders.commons.proto.EAviaConfirmationOutcome;
import ru.yandex.travel.orders.commons.proto.EAviaMqEventOutcome;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;
import ru.yandex.travel.orders.entities.mock.MockAeroflotOrder;
import ru.yandex.travel.orders.repository.mock.MockAeroflotOrderRepository;
import ru.yandex.travel.workflow.exceptions.RetryableException;
import ru.yandex.travel.workflow.exceptions.TooManyRequestsRetryableException;

import static ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceAdapter.checkAvailabilityCheckTestingScenarios;
import static ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceAdapter.checkCreateOrderTestingScenarios;

@RequiredArgsConstructor
@Slf4j
public class MockAeroflotServiceAdapter implements AeroflotService {
    private static final DateTimeFormatter pnrDateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private final MockAeroflotOrderRepository aeroflotRepository;
    private final ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotProviderProperties properties;
    private final TransactionTemplate transactionTemplate;

    @Override
    public boolean checkAvailability(UUID orderId, AeroflotServicePayload payload, TAviaTestContext testContext) {
        String offerId = null;
        try {
            offerId = payload.getVariant().getOffer().getId();
            log.info("Checking availability for offer_id={}", offerId);
            if (properties.isEnableTestingScenarios()) {
                checkAvailabilityCheckTestingScenarios(payload);
            }
            AeroflotVariant token = payload.getVariant();
            if (testContext != null ) {
                var checkAvailabilityOutcome = withTx(orderId, id -> {
                    MockAeroflotOrder mockOrder = aeroflotRepository.findById(id).orElseGet(() -> {
                        var order = new MockAeroflotOrder();
                        order.setId(id);
                        return order;
                    });
                    mockOrder.setCheckAvailabilityResult(testContext.getCheckAvailabilityOutcome());
                    mockOrder.setConfirmationResult(testContext.getConfirmationOutcome());
                    mockOrder.setMqEventResult(testContext.getMqEventOutcome());
                    aeroflotRepository.save(mockOrder);
                    return mockOrder.getCheckAvailabilityResult();
                });
                if (checkAvailabilityOutcome == EAviaCheckAvailabilityOutcome.CAO_PRICE_CHANGED) {
                    Money oldPrice = token.getOffer().getTotalPrice();
                    Money newPrice = oldPrice.add(Money.of(BigDecimal.TEN, oldPrice.getCurrency()));
                    AeroflotTotalOffer offerCopy = token.getOffer().toBuilder()
                            .totalPrice(newPrice)
                            .categoryOffers(token.getOffer().getCategoryOffers().stream()
                                    .peek(catOffer -> {
                                        AeroflotPriceDetail totalPrice = catOffer.getTotalPrice();
                                        Money newTotalPrice = totalPrice.getTotalPrice().add(Money.of(BigDecimal.TEN,
                                                totalPrice.getTotalPrice().getCurrency()));
                                        totalPrice.setTotalPrice(newTotalPrice);
                                        Money newBasePrice = totalPrice.getBasePrice().add(Money.of(BigDecimal.TEN,
                                                totalPrice.getBasePrice().getCurrency()));
                                        totalPrice.setBasePrice(newBasePrice);
                                        catOffer.setTotalPrice(totalPrice);
                                    }).collect(Collectors.toList()))
                            .build();
                    String message = String.format("The offer price has changed; offer_id=%s, old=%s, new=%s",
                            offerId, oldPrice, newPrice);
                    log.info(message);
                    throw new AeroflotOfferPriceChangedException(message, offerCopy);
                }
            }
            log.info("The variant is still available; offer_id={}", offerId);
            return true;
        } catch (VariantNotAvailableException e) {
            log.info("The variant is not available anymore; offer_id={}; e.msg={}", offerId, e.getMessage());
            return false;
        } catch (BookingRetryableException e) {
            throw new RetryableException(e);
        } catch (BookingTooManyRequestsException e) {
            throw new TooManyRequestsRetryableException(e);
        }
    }

    @Override
    public AeroflotOrderCreateResult createOrderAndStartPayment(UUID yaOrderId, AeroflotServicePayload payload,
                                                                String redirectUrl, String tokenizedCard) {
        Preconditions.checkArgument(payload.getTravellers().size() > 0, "Not empty list of travellers id is expected");
        if (properties.isEnableTestingScenarios()) {
            checkCreateOrderTestingScenarios(payload);
        }
        var confirmationResult = withTx(yaOrderId, orderId -> {
            MockAeroflotOrder mockOrder = aeroflotRepository.findById(yaOrderId).orElseThrow();
            return mockOrder.getConfirmationResult();
        });
        if (confirmationResult == null) {
            throw new VariantNotAvailableException("Offer not available as mock aeroflot order not found");
        }
        String pnr = RandomStringUtils.randomAlphabetic(7);
        switch (confirmationResult) {
            case CO_VARIANT_NOT_AVIALABLE:
                throw new VariantNotAvailableException("The offer isn't available anymore: offer_id=" + payload.getVariant().getOffer().getId());
            case CO_PAYMENT_FAILED:
                var orderRef = withTx(yaOrderId, orderId -> {
                    aeroflotRepository.findById(orderId).orElseThrow().setPnr(pnr);
                    return AeroflotOrderRef.builder()
                            .pnr(pnr)
                            .pnrDate("PNR_date_" + LocalDateTime.now().format(pnrDateTimeFormatter))
                            .orderId(yaOrderId.toString())
                            .build();
                });
                throw new AeroflotPaymentException("Payment rejected: offer_id=" + payload.getVariant().getOffer().getId(),
                        PaymentFailureReason.PAYMENT_REJECTED, orderRef);
            case CO_PRICE_CHANGED:
                throw new PriceChangedException("The offer price has changed: offer_id=" + payload.getVariant().getOffer().getId());
            case CO_SUCCESS:
            default:
                var unused = withTx(yaOrderId, orderId -> {
                    MockAeroflotOrder mockOrder = aeroflotRepository.findById(orderId).orElseThrow();
                    mockOrder.setPnr(pnr);
                    if (mockOrder.getMqEventResult() == EAviaMqEventOutcome.MEO_SUCCESS) {
                        mockOrder.setSendMqEventAt(Instant.now().plus(5, ChronoUnit.SECONDS));
                    }
                    return null;
                });
                return AeroflotOrderCreateResult.builder()
                        .statusCode(AeroflotOrderStatus.PAID_TICKETED)
                        .orderRef(AeroflotOrderRef.builder()
                                .pnr(pnr)
                                .pnrDate("PNR_date_" + LocalDateTime.now().format(pnrDateTimeFormatter))
                                .orderId(yaOrderId.toString())
                                .build())
                        .confirmationUrl(redirectUrl)
                        .build();
        }
    }

    @Override
    public AeroflotOrderCreateResult getOrderStatus(UUID yaOrderId, AeroflotServicePayload payload) {
        return withTx(yaOrderId, orderId -> {
            MockAeroflotOrder mockOrder = aeroflotRepository.findById(orderId).orElseThrow();
            AeroflotOrderStatus status = mockOrder.getConfirmationResult() == EAviaConfirmationOutcome.CO_SUCCESS ?
                    AeroflotOrderStatus.PAID_TICKETED : AeroflotOrderStatus.PAYMENT_FAILED;
            return AeroflotOrderCreateResult.builder()
                    .statusCode(status)
                    .orderRef(AeroflotOrderRef.builder()
                            .orderId(yaOrderId.toString())
                            .pnr(mockOrder.getPnr())
                            .build())
                    .build();
        });
    }

    private <ReqT, RspT> RspT withTx(ReqT request, Function<ReqT, RspT> handler) {
        return transactionTemplate.execute((ignored) -> handler.apply(request));
    }
}
