package ru.yandex.travel.orders.workflows.orderitem.bnovo.handlers;

import java.math.RoundingMode;
import java.time.Instant;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.MealData;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableHttpException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableIOException;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.AlreadyCancelledException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.CancelAfterDepartureException;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AdditionalServiceVat;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AdditionalServiceVatBinding;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Booking;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingStatusId;
import ru.yandex.travel.hotels.common.partners.bnovo.model.LegalEntity;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Service;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ServicePriceType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ServiceType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.commons.proto.EVat;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflows.orderitem.bnovo.BNovoConfigurationProperties;
import ru.yandex.travel.orders.workflows.orderitem.bnovo.BnovoFiscalItemCreator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@RequiredArgsConstructor
@Slf4j
public abstract class BaseBNovoHandler extends AnnotatedStatefulWorkflowEventHandler<EBNovoItemState, BNovoOrderItem> {
    protected final Meters meters;
    protected final BNovoConfigurationProperties properties;

    protected CallContext getCallContext(BNovoOrderItem orderItem, CallContext.CallPhase phase) {
        THotelTestContext testContext = null;
        if (orderItem.getTestContext() != null && orderItem.getTestContext() instanceof THotelTestContext) {
            testContext = (THotelTestContext) orderItem.getTestContext();
        }
        return new CallContext(phase, testContext, null, null, orderItem.getHotelItinerary(), null);
    }

    @SuppressWarnings("UnusedReturnValue")
    protected Booking cancelAndGet(BNovoClient client, BNovoHotelItinerary itinerary, boolean force) {
        log.info("Looking up existing reservation");
        var existing = wrap(() -> client.getBookingSync(itinerary.getAccountId(), itinerary.getBNovoNumber()));
        Preconditions.checkState(existing.getOtaBookingId().equals(itinerary.getYandexNumber()),
                "Unexpected yandex number");
        if (existing.getStatusId() == BookingStatusId.CANCELLED) {
            log.info("Booking is already cancelled");
            return existing;
        } else {
            try {
                log.info("Calling cancellation API");
                wrap(() -> client.cancelBookingSync(itinerary.getAccountId(), itinerary.getBNovoNumber(),
                        itinerary.getCustomerEmail()));
            } catch (AlreadyCancelledException ex) {
                log.warn("AlreadyCancelledException occurred on cancellation call");
            } catch (CancelAfterDepartureException ex) {
                if (force) {
                    log.warn("BNovo reports that booking cannot be cancelled due to CancelAfterDepartureException " +
                            "error, but the FORCE flag is present so the error is ignored and the service is " +
                            "considered refunded");
                    return null;
                } else {
                    throw ex;
                }
            }
            log.info("Ensuring that the booking is really cancelled");
            var cancelled = wrap(() -> client
                    .getBookingSync(itinerary.getAccountId(), itinerary.getBNovoNumber()));
            Preconditions.checkState(cancelled.getStatusId() == BookingStatusId.CANCELLED,
                    "Booking state is not Cancelled after the cancellation call");
            return cancelled;
        }
    }

    protected MealData generateMealData(BNovoClient client, BNovoOrderItem item) {
        if (!properties.getBilling().isGenerateMealItems()) {
            return null;
        }
        log.info("Detecting included meals and their prices");

        BNovoHotelItinerary itinerary = item.getItinerary();
        log.info("Calling ratePlans API");
        var reqId = UUID.randomUUID().toString();
        RatePlan plan = wrap(() -> client
                .withCallContext(getCallContext(item, CallContext.CallPhase.ORDER_RESERVATION))
                .getRatePlansSync(itinerary.getAccountId(), reqId)
                .get(itinerary.getBNovoStay().getRates().get(0).getPlanId()));
        Preconditions.checkState(plan != null, "Unable to get rateplan");
        Set<Long> serviceIds = new HashSet<>(plan.getAdditionalServicesIds());
        List<Service> includedMeals =
                wrap(() -> client
                        .withCallContext(getCallContext(item, CallContext.CallPhase.ORDER_RESERVATION))
                        .getServicesSync(itinerary.getAccountId(), reqId).values().stream()
                        .filter(rp -> serviceIds.contains(rp.getId()) && rp.getType() == ServiceType.BOARD && rp.getPrice() > 0)
                        .collect(Collectors.toList())
                );
        var serviceVats = getServiceVats(client, item, itinerary, reqId);
        if (includedMeals.isEmpty()) {
            log.info("No included meal services");
            return null;
        }
        return MealData.builder().items(includedMeals.stream()
                .map(meal -> {
                    Money price = Money.of(meal.getPrice(), itinerary.getFiscalPrice().getCurrency());
                    if (meal.getPriceType() == ServicePriceType.PERSON) {
                        price = price.multiply(itinerary.getOccupancy().getAdults() + itinerary.getOccupancy().getChildren().size());
                    }
                    price = price.multiply(itinerary.getBNovoStay().getNights());
                    var vat = serviceVats.get(meal.getId());
                    return MealData.MealItem.builder()
                            .mealName(meal.getName())
                            .mealPrice(price)
                            .mealVat(vat != null ? mapVat(vat) : item.getAgreement().getVatType().getProtoValue())
                            .build();

                }).collect(Collectors.toList())).build();
    }

    private EVat mapVat(AdditionalServiceVat vat) {
        switch (vat) {
            case VAT0:
                return EVat.VAT_0;
            case VAT10:
                return EVat.VAT_10_110;
            case VAT20:
                return EVat.VAT_20_120;
            case VAT_NONE:
                return EVat.VAT_NONE;
        }
        throw new IllegalArgumentException("Unknown vat");
    }


    private Map<Long, AdditionalServiceVat> getServiceVats(BNovoClient client, BNovoOrderItem item,
                                                           BNovoHotelItinerary itinerary, String reqId) {
        return wrap(() -> {
            LegalEntity ent = null;
            log.info("Getting legal entities");
            Long legalId = client.withCallContext(getCallContext(item, CallContext.CallPhase.ORDER_RESERVATION))
                    .getLegalEntitiesSync(itinerary.getAccountId(), reqId)
                    .getLegalEntities().stream().filter(
                            entity -> entity.getInn().equals(item.getAgreement().getInn()))
                    .map(LegalEntity::getId)
                    .findFirst().orElse(null);
            if (legalId != null) {
                log.info("Getting details on specific legal entity");
                ent = client.withCallContext(getCallContext(item, CallContext.CallPhase.ORDER_RESERVATION))
                        .getLegalEntitySync(itinerary.getAccountId(), legalId, reqId).getLegalEntity();
            }
            if (ent == null) {
                return Collections.emptyMap();
            } else {
                return ent.getAdditionalServicesVat().stream().collect(Collectors.toMap(AdditionalServiceVatBinding::getAdditionalServiceId,
                        AdditionalServiceVatBinding::getVat));
            }
        });
    }

    protected void handleReservedBooking(StateContext<EBNovoItemState, BNovoOrderItem> context,
                                         BNovoHotelItinerary itinerary, Booking booking, BNovoClient client) {
        itinerary.setBNovoNumber(booking.getNumber());
        Preconditions.checkState(booking.getStatusId() == BookingStatusId.CONFIRMED,
                "Unexpected booking status " + booking.getStatusId());
        Preconditions.checkState(!booking.isPaid(),
                "Unexpected booking payment status 'paid': it's just reserved, should not be paid");
        Preconditions.checkState(booking.isBookingGuaranteeAutoBookingCancel(),
                "Unexpected booking auto-cancel status");
        var totalReserved = Money.of(booking.getAmount().setScale(2, RoundingMode.HALF_UP),
                itinerary.getFiscalPrice().getCurrency().getCurrencyCode());
        var totalExpected = itinerary.getFiscalPrice();
        if (totalExpected.compareTo(totalReserved) != 0) {
            log.error("Price mismatch after reservation");
            itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.PRICE_CHANGED));
            context.setState(EBNovoItemState.IS_CANCELLING);
            meters.incrementCancellationCounter(EPartnerId.PI_BNOVO, CancellationDetails.Reason.PRICE_CHANGED);
            context.scheduleEvent(TCancellationCommit.newBuilder().build());
        } else {
            log.info("RESERVED");
            context.setState(EBNovoItemState.IS_RESERVED);
            Instant expiresAt = booking.getOnlineWarrantyDeadlineDate();
            itinerary.setExpiresAtInstant(expiresAt);
            itinerary.setFiscalPrice(totalReserved);
            itinerary.setMealData(generateMealData(client, context.getWorkflowEntity()));
            context.getWorkflowEntity().setExpiresAt(expiresAt);
            BnovoFiscalItemCreator bnovoFiscalItemCreator = new BnovoFiscalItemCreator(properties);
            bnovoFiscalItemCreator.addFiscalItems(context.getWorkflowEntity());
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceReserved.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
        }
    }

    protected <T> T wrap(Supplier<T> supplier) {
        try {
            return supplier.get();
        } catch (PartnerException exception) {
            Throwable cause = exception.getCause();
            if (cause instanceof RetryableHttpException) {
                throw new RetryableException(cause);
            } else if (cause instanceof RetryableIOException) {
                throw new RetryableException(cause);
            } else {
                throw exception;
            }
        }
    }
}
