package ru.yandex.travel.orders.workflows.orderitem.bnovo.handlers;

import java.time.Instant;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TChangeAgreement;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TRefundCommit;
import ru.yandex.travel.orders.workflow.hotels.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflows.orderitem.bnovo.BNovoConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TMoneyAcquired.class})
public class ConfirmedStateHandler extends BaseBNovoHandler {

    private final FinancialEventService financialEventService;
    private final BillingPartnerService billingPartnerService;

    public ConfirmedStateHandler(Meters meters, BNovoConfigurationProperties properties,
                                 FinancialEventService financialEventService,
                                 BillingPartnerService billingPartnerService) {
        super(meters, properties);
        this.financialEventService = financialEventService;
        this.billingPartnerService = billingPartnerService;
    }

    @HandleEvent
    public void handleRefundStart(TRefundStart message, StateContext<EBNovoItemState, BNovoOrderItem> context) {
        context.setState(EBNovoItemState.IS_REFUNDING);
        context.scheduleEvent(TRefundCommit.newBuilder()
                .setToken(message.getToken())
                .setForce(message.getForce())
                .setSkipFinEvents(message.getSkipFinEvents())
                .setMoneyRefundMode(message.getMoneyRefundMode())
                .setReason(message.getReason())
                .setRefundDescription(message.getRefundDescription())
                .build());
        log.info("Scheduled Refund");
    }

    @HandleEvent
    public void handleTChangeAgreement(TChangeAgreement message, StateContext<EBNovoItemState, BNovoOrderItem> context) {
        BNovoOrderItem orderItem = context.getWorkflowEntity();
        log.info("Moving order item from client {} to client {}",
                orderItem.getAgreement().getFinancialClientId(),
                message.getClientId());
        Instant payoutAt = null;
        if (message.hasPayoutDate()) {
            payoutAt = ProtoUtils.toInstant(message.getPayoutDate());
        }
        // TODO(mbobrov): think of better naming and a better place for this call
        financialEventService.registerServiceFullCorrection(orderItem, payoutAt);

        var oldClientId = orderItem.getAgreement().getFinancialClientId();
        var oldContractId = orderItem.getAgreement().getFinancialContractId();

        // Need to update agreement for correct events in registerConfirmedService
        billingPartnerService.checkNewClientForAgreement(orderItem, message.getClientId(), message.getContractId());
        orderItem.getAgreement().setFinancialClientId(message.getClientId());
        orderItem.getAgreement().setFinancialContractId(message.getContractId());

        financialEventService.registerConfirmedService(orderItem, payoutAt);

        // Setting clientId and contractId back as we are afraid to change these values in order
        orderItem.getAgreement().setFinancialClientId(oldClientId);
        orderItem.getAgreement().setFinancialContractId(oldContractId);
    }
}
