package ru.yandex.travel.orders.workflows.orderitem.bnovo.handlers;

import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ConfirmationInfo;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Booking;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingStatusId;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflows.orderitem.bnovo.BNovoConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class ConfirmingStateHandler extends BaseBNovoHandler {
    private final BNovoClient client;

    public ConfirmingStateHandler(BNovoClient client, Meters meters, BNovoConfigurationProperties properties) {
        super(meters, properties);
        this.client = client;
    }

    @HandleEvent
    public void handleConfirmationCommit(TConfirmationCommit message,
                                         StateContext<EBNovoItemState, BNovoOrderItem> context) {

        BNovoHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkState(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(itinerary.getYandexNumber());
        Preconditions.checkNotNull(itinerary.getBNovoNumber());
        Booking confirmedReservation;
        log.info("Looking up existing reservation");
        BNovoClient contextClient = client.withCallContext(getCallContext(context.getWorkflowEntity(),
                CallContext.CallPhase.ORDER_CONFIRMATION));
        var existingReservation = wrap(() -> contextClient
                .getBookingSync(itinerary.getAccountId(), itinerary.getBNovoNumber()));
        Preconditions.checkState(existingReservation.getOtaBookingId().equals(itinerary.getYandexNumber()),
                "Unexpected yandex number");
        if (existingReservation.getStatusId() == BookingStatusId.CANCELLED) {
            CancellationDetails.Reason reason;
            if (existingReservation.isAutomaticallyCancelled()) {
                log.error("Reservation expired");
                reason = CancellationDetails.Reason.HOLD_EXPIRED;
            } else {
                log.error("Reservation cancelled manually by the hotel");
                reason = CancellationDetails.Reason.HOTEL_INTENTION;
            }
            itinerary.setOrderCancellationDetails(CancellationDetails.create(reason));
            context.setState(EBNovoItemState.IS_CANCELLING);
            meters.incrementCancellationCounter(EPartnerId.PI_BNOVO, reason);
            context.scheduleEvent(TCancellationCommit.newBuilder().build());
        } else {
            Preconditions.checkState(existingReservation.getStatusId() == BookingStatusId.CONFIRMED);
            if (existingReservation.isPaid()) {
                log.warn("Attempt to confirm an already confirmed booking, suspecting a retry");
                confirmedReservation = existingReservation;
            } else {
                log.info("Calling confirmation API");
                var confirmationResponse = wrap(() -> contextClient.confirmBookingSync(itinerary.getYandexNumber()));
                var confirmedBookings = confirmationResponse.getConfirmedBookings();
                var cancelledBookings = confirmationResponse.getAlreadyCancelledBookings();
                Preconditions.checkArgument(confirmedBookings.size() + cancelledBookings.size() == 1,
                        "Unexpected number of hotel reservations; confirmed - %s, cancelled - %s",
                        confirmedBookings, cancelledBookings);
                if (confirmationResponse.getAlreadyCancelledBookings().contains(itinerary.getBNovoNumber())) {
                    log.warn("Itinerary is marked as already cancelled while confirming. " +
                            "Scheduling a retry to recheck the status");
                    throw new RetryableException("Itinerary is marked as already cancelled while confirming");
                } else {
                    Preconditions.checkState(confirmationResponse.getConfirmedBookings().contains(itinerary.getBNovoNumber()),
                            "Could not find this booking's number in confirmed bookings list");
                }
                log.info("Reservation confirmed, re-fetching it to verify state");
                confirmedReservation = wrap(() -> contextClient
                        .getBookingSync(itinerary.getAccountId(), itinerary.getBNovoNumber()));
            }
            Preconditions.checkState(confirmedReservation.getStatusId() == BookingStatusId.CONFIRMED,
                    "Unexpected booking status " + confirmedReservation.getStatusId() + " after reservation");
            Preconditions.checkState(confirmedReservation.isPaid(),
                    "Confirmed reservation should have a 'paid' flag");
            Preconditions.checkState(!confirmedReservation.isBookingGuaranteeAutoBookingCancel(),
                    "A confirmed booking should not have a 'booking_guarantee_auto_booking_cancel' flag");
            itinerary.setConfirmation(new ConfirmationInfo());
            itinerary.getConfirmation().setPartnerConfirmationId(itinerary.getBNovoNumber());
            context.setState(EBNovoItemState.IS_CONFIRMED);
            context.getWorkflowEntity().setConfirmedAt(Instant.now());
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceConfirmed.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
            meters.incrementConfirmationCounter(EPartnerId.PI_BNOVO);
            log.info("Confirmation info received, itinerary is CONFIRMED");
        }
    }
}
