package ru.yandex.travel.orders.workflows.orderitem.bnovo.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingList;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TRefresh;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.bnovo.BNovoConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class PollingStateHandler extends BaseBNovoHandler {
    private final BNovoClient client;


    public PollingStateHandler(BNovoClient client, Meters meters, BNovoConfigurationProperties properties) {
        super(meters, properties);
        this.client = client;
    }


    @HandleEvent
    public void handleRefresh(TRefresh message,
                              StateContext<EBNovoItemState, BNovoOrderItem> context) {
        BNovoHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkState(itinerary != null, "No itinerary");
        Preconditions.checkState(itinerary.getYandexNumber() != null, "No yandex number for itinerary");
        log.info("Calling list bookings to search for duplicates");

        BookingList bl = wrap(() -> client
                .withCallContext(getCallContext(context.getWorkflowEntity(),
                        CallContext.CallPhase.ORDER_RESERVATION)).getBookingsSync(itinerary.getAccountId()));
        var existingBooking =
                bl.getBookings().stream().filter(b -> b.getOtaBookingId().equals(itinerary.getYandexNumber())).findAny();
        if (existingBooking.isEmpty()) {
            if (context.getAttempt() >= properties.getMaxDuplicatePollingAttempts()) {
                log.warn("Unable to find duplicate booking, giving up, will cancel the order");
                context.setState(EBNovoItemState.IS_CANCELLED);
                itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.DUPLICATE));
                meters.incrementCancellationCounter(EPartnerId.PI_BNOVO, CancellationDetails.Reason.DUPLICATE);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
            } else {
                String msg = String.format("No bookings with ota_booking_id == '%s' found at partner",
                        itinerary.getYandexNumber());
                log.warn(msg);
                throw new RetryableException(msg);
            }
        } else {
            log.info("Got a booking with ota code == '{}' and partner_code == '{}'",
                    existingBooking.get().getOtaBookingId(),
                    existingBooking.get().getNumber());
            super.handleReservedBooking(context, itinerary, existingBooking.get(), client);
        }
    }
}
