package ru.yandex.travel.orders.workflows.orderitem.bnovo.handlers;

import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;

import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflows.orderitem.bnovo.BNovoConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ReservedStateHandler extends BaseBNovoHandler {
    public ReservedStateHandler(Meters meters, BNovoConfigurationProperties properties) {
        super(meters, properties);
    }

    @HandleEvent
    public void startCancellation(TCancellationStart message,
                                  StateContext<EBNovoItemState, BNovoOrderItem> context) {
        context.setState(EBNovoItemState.IS_CANCELLING);
        context.scheduleEvent(TCancellationCommit.newBuilder().build());
        if (Strings.isNotEmpty(message.getReason())) {
            var itinerary = context.getWorkflowEntity().getItinerary();
            itinerary.setOrderCancellationDetails(CancellationDetails.create(message.getReason()));
            meters.incrementCancellationCounter(EPartnerId.PI_BNOVO, message.getReason());
        }
        log.info("scheduled cancellation");
    }

    @HandleEvent
    public void startGenericCancellation(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart message,
                                         StateContext<EBNovoItemState, BNovoOrderItem> context) {
        startCancellation(TCancellationStart.newBuilder().build(), context);
    }

    @HandleEvent
    public void handleReservationExpired(TReservationExpired message,
                                         StateContext<EBNovoItemState, BNovoOrderItem> context) {
        log.info("Reservation expired. Scheduled cancellation");
        BNovoHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        var reason = CancellationDetails.Reason.MONEY_ACQUIRE_ERROR;
        itinerary.setOrderCancellationDetails(CancellationDetails.create(reason));
        meters.incrementCancellationCounter(EPartnerId.PI_BNOVO, reason);
        context.setState(EBNovoItemState.IS_CANCELLING);
        context.scheduleEvent(TCancellationCommit.newBuilder().build());
    }

    @HandleEvent
    public void startConfirmation(TConfirmationStart message,
                                  StateContext<EBNovoItemState, BNovoOrderItem> context) {
        startGenericConfirmation(null, context);
    }

    @HandleEvent
    public void startGenericConfirmation(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TConfirmationStart message,
                                         StateContext<EBNovoItemState, BNovoOrderItem> context) {
        context.setState(EBNovoItemState.IS_CONFIRMING);
        context.scheduleEvent(TConfirmationCommit.newBuilder().build());
        log.info("scheduled confirmation");
    }
}
