package ru.yandex.travel.orders.workflows.orderitem.bnovo.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.DuplicateBookingException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.SoldOutException;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingJson;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingList;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TRefresh;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TReservationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.bnovo.BNovoConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ReservingStateHandler extends BaseBNovoHandler {
    private final BNovoClient client;

    public ReservingStateHandler(BNovoClient client, Meters meters, BNovoConfigurationProperties properties) {
        super(meters, properties);
        this.client = client;
    }

    @HandleEvent
    public void handleReservationCommit(TReservationCommit message,
                                        StateContext<EBNovoItemState, BNovoOrderItem> context) {
        BNovoHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        itinerary.setServiceId(context.getWorkflowEntity().getId().toString());
        var bookingJson = BookingJson.build(itinerary.getYandexNumber(),
                itinerary.getAccountId(), itinerary.getBNovoStay(),
                itinerary.getGuests().get(0).getFirstName(),
                itinerary.getGuests().get(0).getLastName(),
                itinerary.getCustomerPhone(),
                itinerary.getCustomerEmail(),
                itinerary.getOccupancy() == null ? null : itinerary.getOccupancy().getAdults(),
                itinerary.getOccupancy() == null ? null : itinerary.getOccupancy().getChildren()
        );
        log.info("Calling reservation API");
        try {
            BookingList bookingList = wrap(() -> client
                    .withCallContext(getCallContext(context.getWorkflowEntity(),
                            CallContext.CallPhase.ORDER_RESERVATION))
                    .createBookingSync(itinerary.getAccountId(), bookingJson));
            Preconditions.checkState(bookingList.getBookings().size() == 1,
                    "Unexpected number of hotel reservations");
            var booking = bookingList.getBookings().get(0);
            log.info("Reservation created with BNovo code '{}', Yandex code '{}'", booking.getNumber(),
                    booking.getOtaBookingId());
            handleReservedBooking(context, itinerary, booking, client);
        } catch (SoldOutException e) {
            log.warn("Inventory is sold out", e);
            context.setState(EBNovoItemState.IS_CANCELLED);
            itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.SOLD_OUT));
            meters.incrementCancellationCounter(EPartnerId.PI_BNOVO, CancellationDetails.Reason.SOLD_OUT);
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
        } catch (DuplicateBookingException e) {
            log.warn("Duplicate booking exception has occurred", e);
            context.setState(EBNovoItemState.IS_POLLING_FOR_DUPLICATES);
            context.scheduleEvent(TRefresh.newBuilder().build());
        }
    }
}
