package ru.yandex.travel.orders.workflows.orderitem.bronevik.handlers;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikException;
import ru.yandex.travel.hotels.common.partners.bronevik.CancelOrderFault;
import ru.yandex.travel.hotels.common.partners.bronevik.CancelOrderResponse;
import ru.yandex.travel.hotels.common.partners.bronevik.Child;
import ru.yandex.travel.hotels.common.partners.bronevik.GetHotelOfferPricingFault;
import ru.yandex.travel.hotels.common.partners.bronevik.GetHotelOfferPricingResponse;
import ru.yandex.travel.hotels.common.partners.bronevik.GetOrderResponse;
import ru.yandex.travel.hotels.common.partners.bronevik.Guest;
import ru.yandex.travel.hotels.common.partners.bronevik.OrderService;
import ru.yandex.travel.hotels.common.partners.bronevik.model.FaultCode;
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@RequiredArgsConstructor
@Slf4j
public class BaseBronevikHandler extends AnnotatedStatefulWorkflowEventHandler<EBronevikItemState, BronevikOrderItem> {
    protected final Meters meters;
    protected final BronevikClient bronevikClient;

    protected CallContext getCallContext(StateContext<EBronevikItemState, BronevikOrderItem> context, CallContext.CallPhase phase) {
        THotelTestContext testContext = null;
        BronevikOrderItem orderItem = context.getWorkflowEntity();

        if (orderItem.getTestContext() != null && orderItem.getTestContext() instanceof THotelTestContext) {
            testContext = (THotelTestContext) orderItem.getTestContext();
        }

        return new CallContext(phase, testContext, null, null, orderItem.getHotelItinerary(), null);
    }

    protected boolean checkOfferPricing(StateContext<EBronevikItemState, BronevikOrderItem> context, CallContext callContext) {
        BronevikHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();

        Preconditions.checkNotNull(itinerary.getOfferCode());
        Preconditions.checkNotNull(itinerary.getMeals());
        Preconditions.checkNotNull(itinerary.getCurrency());

        List<Guest> guests = getGuests(itinerary);
        List<Child> children = getChildren(itinerary);
        GetHotelOfferPricingResponse offerPricing = null;
        try {
            offerPricing = bronevikClient.withCallContext(callContext)
                    .getHotelOfferPricingSync(itinerary.getOfferCode(), guests, itinerary.getMeals(),
                            itinerary.getCurrency(), generateRequestId(), children);
        } catch (BronevikException e) {
            Integer errorCode = ((GetHotelOfferPricingFault) e.getCause()).getFaultInfo().getCode();
            FaultCode faultCode = FaultCode.fromValue(errorCode);
            switch (faultCode) {
                case INTERNAL_ERROR:
                    log.warn("Internal Bronevik exception: code {}. Will be retried", errorCode);
                    throw new RetryableException(e);
                case SOLD_OUT:
                    log.info("Sold out");
                    itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.SOLD_OUT));
                    break;
                default:
                    log.error("Unexpected Bronevik exception: code {}", errorCode);
                    itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.INVALID_INPUT));
                    break;
            }
        }

        if (offerPricing == null) {
            return false;
        }


        OrderService orderService = offerPricing.getServices().getService().get(0);
        var currentPrice = getClientPrice(offerPricing.getServices().getService().get(0), itinerary.getCurrency());

        if (!currentPrice.equals(itinerary.getFiscalPrice())) {
            log.warn("Price mismatch: actual {}, expected: {}", currentPrice, itinerary.getFiscalPrice());
            itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.PRICE_CHANGED));

            return false;
        }

        itinerary.setActualPrice(getPartnerPrice(orderService, itinerary.getCurrency()));

        return true;
    }

    protected OrderStatus cancelOrderWithStatusCheck(BronevikHotelItinerary itinerary, CallContext callContext) {
        GetOrderResponse order = bronevikClient.withCallContext(callContext)
                .getOrderSync(itinerary.getOrderId(), generateRequestId());
        int orderStatusId = order.getOrder().getServices().getService().get(0).getStatusId();
        OrderStatus status = OrderStatus.fromValue(orderStatusId);

        if (status == OrderStatus.NOT_CONFIRMED
                || status == OrderStatus.CANCELLED_WITH_PENALTY
                || status == OrderStatus.CANCELLED_WITHOUT_PENALTY
                || status == OrderStatus.AWAITING_CANCELLATION
                || status == OrderStatus.CANCELLATION_IS_REQUESTED) {
            return status;
        }

        return cancelOrder(itinerary, callContext);
    }

    private OrderStatus cancelOrder(BronevikHotelItinerary itinerary, CallContext callContext) {
        CancelOrderResponse cancelResult = null;

        try {
            cancelResult = bronevikClient.withCallContext(callContext)
                    .cancelOrderSync(itinerary.getOrderId(), generateRequestId());
        } catch (BronevikException e) {
            if (Objects.equals(((CancelOrderFault) e.getCause()).getFaultInfo().getCode(), FaultCode.INTERNAL_ERROR.getValue())) {
                throw new RetryableException(e);
            }
        }

        Preconditions.checkState(cancelResult.isResult(), "Failed to cancel order. Order can not be cancelled in NOT_CONFIRMED or CANCELLED status");

        GetOrderResponse order = bronevikClient.withCallContext(callContext)
                .getOrderSync(itinerary.getOrderId(), generateRequestId());
        int orderStatusId = order.getOrder().getServices().getService().get(0).getStatusId();

        return OrderStatus.fromValue(orderStatusId);
    }

    protected void toCancelledStatus(StateContext<EBronevikItemState, BronevikOrderItem> context) {
        context.setState(EBronevikItemState.IS_CANCELLED);
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
        log.info("CANCELLED");
    }

    protected List<Child> getChildren(BronevikHotelItinerary itinerary) {
        Map<Integer, Integer> children = new HashMap<>();
        itinerary.getGuests().forEach(guest -> {
            if (guest.isChild()) {
                children.compute(guest.getAge(), (key, oldValue) -> oldValue == null ? 1 : oldValue + 1);
            }
        });
        return children.keySet().stream().map(key -> {
            var child = new Child();
            child.setAge(key);
            child.setCount(children.get(key));
            return child;
        }).collect(Collectors.toList());
    }

    protected List<Guest> getGuests(BronevikHotelItinerary itinerary) {
        var firstGuest = itinerary.getGuests().get(0);

        Preconditions.checkState(firstGuest.hasFilledName());

        return itinerary.getGuests().stream().map(offerGuest -> {
            var bronevikGuest = new Guest();

            if (offerGuest.hasFilledName()) {
                bronevikGuest.setFirstName(offerGuest.getFirstName());
                bronevikGuest.setLastName(offerGuest.getLastName());
            } else {
                bronevikGuest.setFirstName(firstGuest.getFirstName());
                bronevikGuest.setLastName(firstGuest.getLastName());
            }

            return bronevikGuest;
        }).collect(Collectors.toList());
    }

    protected String generateRequestId() {
        return UUID.randomUUID().toString();
    }

    protected Money getClientPrice(OrderService service, String currency) {
        var priceDetails = service.getPriceDetails().getClient().getClientCurrency();

        return Money.of(
                BigDecimal.valueOf(priceDetails.getGross().getPrice()).setScale(0, RoundingMode.HALF_UP),
                currency);
    }

    protected Money getPartnerPrice(OrderService service, String currency) {
        var priceDetails = service.getPriceDetails().getClient().getClientCurrency();

        return Money.of(
                BigDecimal.valueOf(priceDetails.getGross().getPrice()).setScale(2, RoundingMode.HALF_UP),
                currency);
    }
}
