package ru.yandex.travel.orders.workflows.orderitem.bronevik.handlers;

import java.time.Instant;
import java.util.List;
import java.util.Objects;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ConfirmationInfo;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikException;
import ru.yandex.travel.hotels.common.partners.bronevik.Child;
import ru.yandex.travel.hotels.common.partners.bronevik.CreateOrderFault;
import ru.yandex.travel.hotels.common.partners.bronevik.CreateOrderResponse;
import ru.yandex.travel.hotels.common.partners.bronevik.Guest;
import ru.yandex.travel.hotels.common.partners.bronevik.Order;
import ru.yandex.travel.hotels.common.partners.bronevik.OrderServiceAccommodation;
import ru.yandex.travel.hotels.common.partners.bronevik.SearchOrdersFault;
import ru.yandex.travel.hotels.common.partners.bronevik.SearchOrdersResponse;
import ru.yandex.travel.hotels.common.partners.bronevik.model.FaultCode;
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class ConfirmingStateHandler extends BaseBronevikHandler {

    public ConfirmingStateHandler(Meters meters, BronevikClient bronevikClient) {
        super(meters, bronevikClient);
    }

    @HandleEvent
    public void handleConfirmationCommit(TConfirmationCommit message,
                                         StateContext<EBronevikItemState, BronevikOrderItem> context) {
        BronevikHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();

        Order order = searchOrders(itinerary, getCallContext(context, CallContext.CallPhase.ORDER_CREATION));

        if (order != null) {
            confirmOrder(order, context);

            return;
        }

        boolean isActualOffer = checkOfferPricing(context, getCallContext(context, CallContext.CallPhase.ORDER_CREATION));

        if (!isActualOffer) {
            toCancelledStatus(context);

            return;
        }

        order = createOrder(itinerary, context);

        if (order == null) {
            toCancelledStatus(context);

            return;
        }

        confirmOrder(order, context);
    }

    private Order searchOrders(BronevikHotelItinerary itinerary, CallContext callContext) {
        SearchOrdersResponse orders;

        try {
            orders = bronevikClient.withCallContext(callContext)
                    .searchOrdersByReferenceIdSync(itinerary.getYandexNumber(), generateRequestId());
        } catch (BronevikException e) {
            Integer errorCode = ((SearchOrdersFault)e.getCause()).getFaultInfo().getCode();

            if (Objects.equals(errorCode, FaultCode.INTERNAL_ERROR.getValue())) {
                log.warn("Internal Bronevik exception: code {}. Will be retried", errorCode);
                throw new RetryableException(e);
            }

            log.error("Unexpected Bronevik exception: code {}", errorCode);

            throw new RuntimeException(e);
        }

        if (orders.getOrders().getOrder().isEmpty()) {
            return null;
        }

        return orders.getOrders().getOrder().stream().filter(order -> {
            OrderServiceAccommodation service = (OrderServiceAccommodation) order.getServices().getService().get(0);

            return Objects.equals(service.getOfferCode(), itinerary.getOfferCode());
        }).findFirst().orElse(null);
    }

    private Order createOrder(BronevikHotelItinerary itinerary, StateContext<EBronevikItemState, BronevikOrderItem> context) {
        List<Guest> guests = getGuests(itinerary);
        List<Child> children = getChildren(itinerary);

        Order order = null;
        try {
            CreateOrderResponse orderResponse = bronevikClient.withCallContext(getCallContext(context, CallContext.CallPhase.ORDER_CREATION))
                    .createOrderSync(itinerary.getOfferCode(), guests, itinerary.getMeals(), itinerary.getCurrency(),
                            itinerary.getYandexNumber(), generateRequestId(), children);
            order = orderResponse.getOrder();
        } catch (BronevikException e) {
            Integer errorCode = ((CreateOrderFault) e.getCause()).getFaultInfo().getCode();
            FaultCode faultCode = FaultCode.fromValue(errorCode);
            switch (faultCode) {
                case INTERNAL_ERROR:
                    log.warn("Internal Bronevik exception: code {}. Will be retried", errorCode);
                    throw new RetryableException(e);
                case SOLD_OUT:
                    log.info("Sold out");
                    itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.SOLD_OUT));
                    break;
                default:
                    log.error("Unexpected Bronevik exception: code {}", errorCode);
                    itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.INVALID_INPUT));
                    break;
            }
        }

        return order;
    }

    private void confirmOrder(Order order, StateContext<EBronevikItemState, BronevikOrderItem> context) {
        BronevikHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        int statusId = order.getServices().getService().get(0).getStatusId();
        OrderStatus status = OrderStatus.fromValue(statusId);
        Integer bronevikOrderId = order.getId();

        if (status == OrderStatus.NOT_CONFIRMED) {
            log.info("Order in manual confirmation not confirmed by partner");

            toCancelledStatus(context);

            return;
        }

        if (status != OrderStatus.CONFIRMED) {
            log.info("Order with manual confirmation or with unknown status");

            context.setState(EBronevikItemState.IS_CANCELLING);
            context.scheduleEvent(TCancellationCommit.newBuilder().build());

            return;
        }

        log.info("Order confirmed");
        itinerary.setOrderId(bronevikOrderId);

        var confirmationInfo = new ConfirmationInfo();
        confirmationInfo.setPartnerConfirmationId(bronevikOrderId.toString());
        itinerary.setConfirmation(confirmationInfo);

        context.setState(EBronevikItemState.IS_CONFIRMED);
        meters.incrementConfirmationCounter(EPartnerId.PI_BRONEVIK);
        context.getWorkflowEntity().setConfirmedAt(Instant.now());
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceConfirmed.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());

        log.info("Confirmation info received, itinerary is CONFIRMED");
    }

    private boolean checkPriceChange(BronevikHotelItinerary itinerary, Order order) {
        var clientPrice = getClientPrice(order.getServices().getService().get(0), itinerary.getCurrency());

        log.info("Compare client price from order {} and fiscal price {}", clientPrice, itinerary.getFiscalPrice());

        return !clientPrice.equals(itinerary.getFiscalPrice());
    }
}
