package ru.yandex.travel.orders.workflows.orderitem.bronevik.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient;
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.proto.THotelRefundToken;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.TRefundCommit;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class RefundingStateHandler extends BaseBronevikHandler {

    public RefundingStateHandler(Meters meters, BronevikClient bronevikClient) {
        super(meters, bronevikClient);
    }

    @HandleEvent
    public void handleRefundCommit(TRefundCommit message, StateContext<EBronevikItemState, BronevikOrderItem> context) {
        BronevikHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();

        Preconditions.checkNotNull(itinerary.getOrderId());

        OrderStatus status = cancelOrderWithStatusCheck(itinerary, getCallContext(context, CallContext.CallPhase.ORDER_CANCELLATION));
        THotelRefundToken refundToken = RefundingUtils.getRefundToken(message.getToken());

        if (refundToken.getPenaltyAmount().getAmount() > 0 && status == OrderStatus.CANCELLED_WITHOUT_PENALTY) {
            log.warn("Bronevik has canceled order without penalty, but hotel order is not fully refunded");
        }

        if (refundToken.getPenaltyAmount().getAmount() == 0 && status == OrderStatus.CANCELLED_WITH_PENALTY) {
            log.warn("Bronevik has canceled order with penalty, but hotel order is fully refunded");
        }

        itinerary.setRefundInfo(RefundingUtils.createRefundInfo(refundToken, message.getReason()));
        context.setState(EBronevikItemState.IS_REFUNDED);
        RefundingUtils.scheduleServiceRefundedEvent(context, refundToken,
                message.getSkipFinEvents(), message.getMoneyRefundMode(), message.getRefundDescription());
        log.info("REFUNDED");
    }
}
