package ru.yandex.travel.orders.workflows.orderitem.bus;

import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.repository.BusTicketRefundRepository;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.services.buses.BusPartnersProperties;
import ru.yandex.travel.orders.services.buses.BusPartnersService;
import ru.yandex.travel.orders.services.buses.BusesServiceProvider;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.CalculatingFeeStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.CancelledStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.CancellingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.ConfirmedStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.ConfirmingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.NewStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.RefundingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.ReservedStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.bus.handlers.ReservingStateHandler;
import ru.yandex.travel.workflow.StatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.base.ProxyStateMapWorkflowEventHandler;
import ru.yandex.travel.workflow.repository.EntityStateTransitionRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties({BusProperties.class, BusPartnersProperties.class})
public class BusWorkflowConfiguration {
    private final EntityStateTransitionRepository entityStateTransitionRepository;

    private final BusesServiceProvider busesServiceProvider;
    private final BillingPartnerService billingPartnerService;
    private final OrderRefundRepository orderRefundRepository;
    private final BusTicketRefundRepository ticketRefundRepository;
    private final WorkflowRepository workflowRepository;
    private final BusPartnersService busPartnersService;
    private final BusProperties busProperties;

    @Bean
    public WorkflowEventHandler<BusOrderItem> busOrderItemWorkflowEventHandler() {
        ImmutableMap.Builder<EOrderItemState, StatefulWorkflowEventHandler<EOrderItemState, BusOrderItem>> builder = ImmutableMap.builder();
        builder
                .put(EOrderItemState.IS_NEW, new NewStateHandler(billingPartnerService, busPartnersService))
                .put(EOrderItemState.IS_RESERVING, new ReservingStateHandler(busesServiceProvider, busProperties))
                .put(EOrderItemState.IS_CALCULATING_FEE_BUS, new CalculatingFeeStateHandler(busPartnersService, busProperties))
                .put(EOrderItemState.IS_RESERVED, new ReservedStateHandler())
                .put(EOrderItemState.IS_CANCELLING, new CancellingStateHandler(busesServiceProvider, busProperties))
                .put(EOrderItemState.IS_CANCELLED, new CancelledStateHandler())
                .put(EOrderItemState.IS_CONFIRMING, new ConfirmingStateHandler(busesServiceProvider, busProperties))
                .put(EOrderItemState.IS_CONFIRMED, new ConfirmedStateHandler(orderRefundRepository, ticketRefundRepository, workflowRepository))
                .put(EOrderItemState.IS_REFUNDING, new RefundingStateHandler());
        return new ProxyStateMapWorkflowEventHandler<>(builder.build(), entityStateTransitionRepository);
    }
}
