package ru.yandex.travel.orders.workflows.orderitem.bus;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.env.Environment;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.bus.service.BusesServiceConfiguration;
import ru.yandex.travel.bus.service.BusesServiceExpConfiguration;
import ru.yandex.travel.orders.commons.proto.TBusTestContext;
import ru.yandex.travel.orders.repository.mock.MockBusOrderRepository;
import ru.yandex.travel.orders.services.buses.BusesServiceProvider;
import ru.yandex.travel.orders.services.mock.MockBusesService;

@Configuration
@Import({
        BusesServiceConfiguration.class,
        BusesServiceExpConfiguration.class,
})
@RequiredArgsConstructor
public class BusesServiceProviderConfiguration {
    private final Environment environment;

    @Bean
    @ConditionalOnProperty(value = "buses-service.mock-client-enabled")
    public MockBusesService mockBusesService(MockBusOrderRepository mockBusOrderRepository,
                                             PlatformTransactionManager transactionManager) {
        if (environment.acceptsProfiles("prod")) {
            throw new RuntimeException("\n\nATTENTION: MOCK BUS CLIENT MUST BE DISABLED FOR PROD ENVIRONMENT\n\n");
        }
        DefaultTransactionDefinition defaultTransactionDefinition = new DefaultTransactionDefinition();
        defaultTransactionDefinition.setName("MockBusesWorkerClient");
        defaultTransactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        defaultTransactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition transactionDefinition = new DefaultTransactionDefinition(defaultTransactionDefinition);
        TransactionTemplate transactionTemplate = new TransactionTemplate(transactionManager, transactionDefinition);
        return new MockBusesService(mockBusOrderRepository, transactionTemplate);
    }

    @Bean
    public BusesServiceProvider busesServiceProvider(BusesService busesService,
                                                     @Qualifier("busesServiceExp") BusesService busesServiceExp,
                                                     @Autowired(required = false) MockBusesService mockBusesService) {
        return orderItem -> {
            if (orderItem != null) {
                var testContext = orderItem.getTestContext();
                if (testContext instanceof TBusTestContext) {
                    return mockBusesService;
                } else {
                    boolean useExp = orderItem.getOrder().getKVExperiments().stream().anyMatch(
                            kv -> kv.getKey().equals("BUSES_use_exp_backend"));
                    if (useExp) {
                        return busesServiceExp;
                    }
                    return busesService;
                }
            }
            return busesService;
        };
    }

}
