package ru.yandex.travel.orders.workflows.orderitem.bus.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.bus.model.BusPartner;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.services.FiscalTitleGenerator;
import ru.yandex.travel.orders.services.buses.BusPartnersService;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TFeeCalculationCommit;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class CalculatingFeeStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, BusOrderItem> {
    private final BusPartnersService busPartnersService;
    private final BusProperties busProperties;

    @HandleEvent
    public void handleFeeCalculationCommit(TFeeCalculationCommit event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        BusReservation payload = orderItem.getPayload();
        for (BusesTicket t : payload.getOrder().getTickets()) {
            Money yandexFee = busPartnersService.calculateYandexFee(payload.getRide().getBusPartner(), t.getPrice());
            t.setYandexFee(yandexFee);
        }
        generateFiscalItems(orderItem, ctx);
        ctx.setState(EOrderItemState.IS_RESERVED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), TServiceReserved.newBuilder()
                .setServiceId(orderItem.getId().toString()).build());
    }

    private void generateFiscalItems(BusOrderItem orderItem, StateContext<EOrderItemState, BusOrderItem> ctx) {
        var fiscalCounter = 0;
        BusRide ride = orderItem.getPayload().getRide();
        BusPartner partner = ride.getBusPartner();
        for (BusesTicket ticket : orderItem.getPayload().getOrder().getTickets()) {
            Money ticketPrice = ticket.getTicketPrice();
            if (ticketPrice.isPositive()) {
                ticket.setTicketFiscalItemInternalId(fiscalCounter++);
                var fiscalItem = new FiscalItem();
                fiscalItem.setInternalId(ticket.getTicketFiscalItemInternalId());
                fiscalItem.setMoneyAmount(ticketPrice);
                fiscalItem.setVatType(VatType.fromEVat(ticket.getTicketVat().getTravelProtoValue()));
                fiscalItem.setTitle(FiscalTitleGenerator.makeBusFiscalItemTitle(
                        busProperties.getTicketFiscalTitle(), ticket.getPassenger().getSeatPartnerId()));
                fiscalItem.setType(busPartnersService.getTicketFiscalItemType(partner));
                fiscalItem.setInn(ride.getSupplier().getTaxationNumber());
                orderItem.addFiscalItem(fiscalItem);
            }
            if (ticket.getPartnerFee().isPositive()) {
                ticket.setPartnerFeeFiscalItemInternalId(fiscalCounter++);
                var fiscalItem = new FiscalItem();
                fiscalItem.setInternalId(ticket.getPartnerFeeFiscalItemInternalId());
                fiscalItem.setMoneyAmount(ticket.getPartnerFee());
                fiscalItem.setVatType(VatType.fromEVat(ticket.getPartnerFeeVat().getTravelProtoValue()));
                fiscalItem.setTitle(FiscalTitleGenerator.makeBusFiscalItemTitle(
                        busProperties.getPartnerFeeFiscalTitle(), ticket.getPassenger().getSeatPartnerId()));
                fiscalItem.setType(busPartnersService.getPartnerFeeFiscalItemType(partner));
                fiscalItem.setInn(ride.getSupplier().getTaxationNumber());
                orderItem.addFiscalItem(fiscalItem);
            }
            if (ticket.getYandexFee().isPositive()) {
                ticket.setYandexFeeFiscalItemInternalId(fiscalCounter++);
                var fiscalItem = new FiscalItem();
                fiscalItem.setInternalId(ticket.getYandexFeeFiscalItemInternalId());
                fiscalItem.setMoneyAmount(ticket.getYandexFee());
                fiscalItem.setVatType(VatType.fromEVat(busProperties.getYandexFeeVat()));
                fiscalItem.setTitle(FiscalTitleGenerator.makeBusFiscalItemTitle(
                        busProperties.getYandexFeeFiscalTitle(), ticket.getPassenger().getSeatPartnerId()));
                fiscalItem.setType(busPartnersService.getYandexFeeFiscalItemType(partner));
                orderItem.addFiscalItem(fiscalItem);
            }
        }
    }
}
