package ru.yandex.travel.orders.workflows.orderitem.bus.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.bus.service.BusesServiceRetryableException;
import ru.yandex.travel.buses.backend.proto.worker.TCancelBookingRequest;
import ru.yandex.travel.buses.backend.proto.worker.TRequestHeader;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.services.buses.BusesServiceProvider;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {TCancellationStart.class, TReservationExpired.class})
public class CancellingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, BusOrderItem> {
    private final BusesServiceProvider busesServiceProvider;
    private final BusProperties busProperties;

    @HandleEvent
    public void handleCancellationCommit(TCancellationCommit event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        BusReservation payload = orderItem.getPayload();
        TCancelBookingRequest request = TCancelBookingRequest
                .newBuilder()
                .setHeader(TRequestHeader.newBuilder())
                .setSupplierId(payload.getRide().getSupplierId())
                .setOrderId(payload.getOrder().getId())
                .build();
        BusesService busesService = busesServiceProvider.getBusesServiceByOrderItem(orderItem);
        try {
            busesService.cancelBooking(request);
        } catch (Exception e) {
            if (e instanceof BusesServiceRetryableException &&
                    ctx.getAttempt() < busProperties.getCancellationMaxTries()) {
                throw new RetryableException(e, busProperties.getCancellationRetryDelay());
            }
            log.warn("Booking cancellation error: ", e);
        }

        ctx.setState(EOrderItemState.IS_CANCELLED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), TServiceCancelled.newBuilder()
                .setServiceId(orderItem.getId().toString()).build());
    }
}
