package ru.yandex.travel.orders.workflows.orderitem.bus.handlers;

import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.BusTicketRefund;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.repository.BusTicketRefundRepository;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TBusRefundTicket;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TRefundTicketStart;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Slf4j
@RequiredArgsConstructor
public class ConfirmedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, BusOrderItem> {
    private final OrderRefundRepository orderRefundRepository;
    private final BusTicketRefundRepository ticketRefundRepository;
    private final WorkflowRepository workflowRepository;

    @HandleEvent
    public void handleRefundStart(TRefundStart event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        OrderRefund orderRefund = orderRefundRepository.getOne(UUID.fromString(event.getOrderRefundId()));
        Preconditions.checkArgument(event.getToken().getTicketsList().size() > 0, "Tickets to refund can not be empty");
        Map<String, BusesTicket> ticketsById = orderItem.getPayload().getOrder().getTickets().stream()
                .collect(Collectors.toMap(BusesTicket::getId, t -> t));
        for (TBusRefundTicket refundTicket : event.getToken().getTicketsList()) {
            BusesTicket ticket = ticketsById.get(refundTicket.getTicketId());
            Preconditions.checkArgument(ticket != null, "Not found ticket by id %s", refundTicket.getTicketId());
            Preconditions.checkState(ticket.getStatus() == BusTicketStatus.SOLD,
                    "Ticket to refund with id %s has status %s", ticket.getId(), ticket.getStatus());
            BusTicketRefund ticketRefund = BusTicketRefund.createRefund(orderItem, refundTicket, orderRefund);
            Workflow workflow = Workflow.createWorkflowForEntity(ticketRefund, ctx.getWorkflowEntity());
            workflowRepository.save(workflow);
            ticketRefundRepository.save(ticketRefund);
        }
        ctx.setState(EOrderItemState.IS_REFUNDING);
        ctx.scheduleEvent(TRefundTicketStart.newBuilder().build());
    }
}
