package ru.yandex.travel.orders.workflows.orderitem.bus.handlers;

import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.service.BusesModelsHelpers;
import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.bus.service.BusesServiceRetryableException;
import ru.yandex.travel.buses.backend.proto.worker.EOrderStatus;
import ru.yandex.travel.buses.backend.proto.worker.TConfirmRequest;
import ru.yandex.travel.buses.backend.proto.worker.TConfirmResponse;
import ru.yandex.travel.buses.backend.proto.worker.TRequestHeader;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.services.buses.BusesServiceProvider;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
public class ConfirmingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, BusOrderItem> {
    private final BusesServiceProvider busesServiceProvider;
    private final BusProperties busProperties;

    @HandleEvent
    public void handleConfirmationCommit(TConfirmationCommit event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        BusReservation payload = orderItem.getPayload();
        TConfirmRequest request = TConfirmRequest
                .newBuilder()
                .setHeader(TRequestHeader.newBuilder())
                .setSupplierId(orderItem.getPayload().getRide().getSupplierId())
                .setOrderId(orderItem.getPayload().getOrder().getId())
                .build();
        TConfirmResponse response;
        BusesService busesService = busesServiceProvider.getBusesServiceByOrderItem(orderItem);
        try {
            response = busesService.confirm(request);
        } catch (Exception e) {
            if (e instanceof BusesServiceRetryableException &&
                    ctx.getAttempt() < busProperties.getConfirmationMaxTries()) {
                throw new RetryableException(e, busProperties.getConfirmationRetryDelay());
            }
            log.warn("Exception when confirming: ", e);
            ctx.setState(EOrderItemState.IS_CANCELLED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), TServiceCancelled.newBuilder()
                    .setServiceId(orderItem.getId().toString()).build());
            return;
        }
        if (response.getOrder().getStatus() == EOrderStatus.OS_CANCELLED) {
            log.warn("Unexpected confirmed order state: OS_CANCELLED");
            ctx.setState(EOrderItemState.IS_CANCELLED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), TServiceCancelled.newBuilder()
                    .setServiceId(orderItem.getId().toString()).build());
            return;
        }
        Preconditions.checkState(response.getOrder().getStatus() == EOrderStatus.OS_SOLD,
                "Unexpected confirmed order state: %s", response.getOrder().getStatus());

        BusesModelsHelpers.migrateTOrderToBusesOrder(response.getOrder(), payload.getOrder());
        orderItem.setConfirmedAt(Instant.now());
        if (orderItem.getOrder() != null) {
            orderItem.getOrder().getInvoices().forEach(i -> i.setClearAt(Instant.now()));
        }
        ctx.setState(EOrderItemState.IS_CONFIRMED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), TServiceConfirmed.newBuilder()
                .setServiceId(orderItem.getId().toString()).build());
    }
}
