package ru.yandex.travel.orders.workflows.orderitem.bus.handlers;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.BusTicketRefund;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefundFailed;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TRefundTicketStart;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.EBusTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.TStartRefund;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.TTicketRefundFailed;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.TTicketRefundSuccess;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsToCardOnlyMarkup;

@Slf4j
@RequiredArgsConstructor
public class RefundingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, BusOrderItem> {

    @HandleEvent
    public void handleRefundTicketStart(TRefundTicketStart event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        BusTicketRefund ticketRefundToDo = getTicketRefundToDo(orderItem);
        Preconditions.checkState(ticketRefundToDo != null, "There is no ticket refund to start");
        ctx.scheduleExternalEvent(ticketRefundToDo.getWorkflow().getId(), TStartRefund.newBuilder().build());
    }

    private BusTicketRefund getTicketRefundToDo(BusOrderItem orderItem) {
        return orderItem.getBusTicketRefunds().stream()
                .filter(x -> x.getState() == EBusTicketRefundState.RS_NEW)
                .findFirst().orElse(null);
    }

    @HandleEvent
    public void handleTicketRefundSuccess(TTicketRefundSuccess event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();

        BusesTicket ticket = orderItem.getPayload().getOrder().getTickets().stream()
                .filter(t -> t.getId().equals(event.getTicketId()))
                .findFirst().orElseThrow();
        ticket.setStatus(BusTicketStatus.RETURNED);
        ticket.setRefundedPrice(ProtoUtils.fromTPrice(event.getPrice()));

        BusTicketRefund ticketRefundToDo = getTicketRefundToDo(orderItem);
        if (ticketRefundToDo != null) {
            ctx.scheduleExternalEvent(ticketRefundToDo.getWorkflow().getId(), TStartRefund.newBuilder().build());
            return;
        }
        onRefundComplete(UUID.fromString(event.getRefundId()), ctx);
    }

    @HandleEvent
    public void handleTicketRefundFailed(TTicketRefundFailed event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        BusTicketRefund ticketRefundToDo = getTicketRefundToDo(orderItem);
        if (ticketRefundToDo != null) {
            ctx.scheduleExternalEvent(ticketRefundToDo.getWorkflow().getId(), TStartRefund.newBuilder().build());
            return;
        }
        onRefundComplete(UUID.fromString(event.getRefundId()), ctx);
    }

    private void onRefundComplete(UUID refundId, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        EOrderItemState orderItemState = checkOrderItemState(orderItem);

        GenericOrderUserRefund orderRefund = (GenericOrderUserRefund)orderItem.getOrder().getOrderRefunds().stream()
                .filter(x -> refundId.equals(x.getId())).findFirst().orElseThrow();
        List<BusTicketRefund> successTicketRefunds = orderRefund.getBusTicketRefunds().stream()
                .filter(x -> x.getOrderItem().getId().equals(orderItem.getId()))
                .filter(x -> x.getState() == EBusTicketRefundState.RS_REFUNDED)
                .collect(Collectors.toList());

        Message orderMessage;
        if (successTicketRefunds.size() > 0) {
            Map<Long, Money> targetFiscalItems = createRefundFiscalItems(successTicketRefunds, orderItem);
            Map<Long, MoneyMarkup> targetFiscalItemsMarkup = convertTargetFiscalItemsToCardOnlyMarkup(targetFiscalItems);
            orderMessage = TServiceRefunded.newBuilder()
                    .setServiceId(orderItem.getId().toString())
                    .setOrderRefundId(orderRefund.getId().toString())
                    .putAllTargetFiscalItems(RefundingUtils.convertTargetFiscalItemsToProto(targetFiscalItems))
                    .putAllTargetFiscalItemsMarkup(RefundingUtils.convertTargetFiscalItemsMarkupToProto(targetFiscalItemsMarkup))
                    .setServiceConfirmed(orderItemState == EOrderItemState.IS_CONFIRMED)
                    .build();
        } else {
            orderMessage = TServiceRefundFailed.newBuilder()
                    .setServiceId(orderItem.getId().toString())
                    .setOrderRefundId(orderRefund.getId().toString())
                    .build();
        }
        if (orderItemState == EOrderItemState.IS_REFUNDED) {
            orderItem.setRefundedAt(Instant.now());
        }
        ctx.setState(orderItemState);
        ctx.scheduleExternalEvent(orderItem.getOrderWorkflowId(), orderMessage);
    }

    private EOrderItemState checkOrderItemState(BusOrderItem orderItem) {
        if (orderItem.getPayload().getOrder().getTickets().stream()
                .allMatch(x -> x.getStatus() == BusTicketStatus.RETURNED))
            return EOrderItemState.IS_REFUNDED;
        else
            return EOrderItemState.IS_CONFIRMED;
    }

    private static Map<Long, Money> createRefundFiscalItems(List<BusTicketRefund> successTicketRefunds, BusOrderItem orderItem) {
        Map<Integer, Long> fiscalItemIdByInternalId = orderItem.getFiscalItems().stream()
                .collect(Collectors.toMap(FiscalItem::getInternalId, FiscalItem::getId));
        Map<String, BusesTicket> ticketsById = orderItem.getPayload().getOrder().getTickets().stream()
                .collect(Collectors.toMap(BusesTicket::getId, x -> x));
        Map<Long, Money> targetFiscalItems = new HashMap<>();
        for (BusTicketRefund ticketRefund : successTicketRefunds) {
            BusesTicket ticket = ticketsById.get(ticketRefund.getPayload().getTicketId());
            Money priceWithoutFee = ticket.getPrice();
            Money ticketPrice = ticket.getTicketPrice();
            Money partnerFee = ticket.getPartnerFee();
            Money refundAmount = ticketRefund.getPayload().getRefundAmount();
            // yandex fee can be refunded only by manual refund
            Preconditions.checkState(refundAmount.isLessThanOrEqualTo(priceWithoutFee), "Can not refund more than price");
            if (refundAmount.isGreaterThan(ticketPrice)) {
                targetFiscalItems.put(
                        fiscalItemIdByInternalId.get(ticket.getTicketFiscalItemInternalId()),
                        Money.zero(ticketPrice.getCurrency()));
                refundAmount = refundAmount.subtract(ticketPrice);
                targetFiscalItems.put(
                        fiscalItemIdByInternalId.get(ticket.getPartnerFeeFiscalItemInternalId()),
                        partnerFee.subtract(refundAmount));
            } else if (!refundAmount.isZero()) {
                targetFiscalItems.put(
                        fiscalItemIdByInternalId.get(ticket.getTicketFiscalItemInternalId()),
                        ticketPrice.subtract(refundAmount));
            }
        }
        return targetFiscalItems;
    }
}
