package ru.yandex.travel.orders.workflows.orderitem.bus.handlers;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusesPassenger;
import ru.yandex.travel.bus.service.BusesModelsHelpers;
import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.bus.service.BusesServiceRetryableException;
import ru.yandex.travel.buses.backend.proto.worker.EOrderStatus;
import ru.yandex.travel.buses.backend.proto.worker.TBookContacts;
import ru.yandex.travel.buses.backend.proto.worker.TBookRequest;
import ru.yandex.travel.buses.backend.proto.worker.TBookResponse;
import ru.yandex.travel.buses.backend.proto.worker.TRequestHeader;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.services.buses.BusesServiceProvider;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TFeeCalculationCommit;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TReservationCommit;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
public class ReservingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, BusOrderItem> {
    private final BusesServiceProvider busesServiceProvider;
    private final BusProperties busProperties;

    @HandleEvent
    public void handleReservationCommit(TReservationCommit event, StateContext<EOrderItemState, BusOrderItem> ctx) {
        BusOrderItem orderItem = ctx.getWorkflowEntity();
        BusReservation payload = orderItem.getPayload();
        TBookRequest.Builder requestBuilder = TBookRequest
                .newBuilder()
                .setHeader(TRequestHeader.newBuilder())
                .setSupplierId(payload.getRide().getSupplierId())
                .setContacts(TBookContacts.newBuilder()
                        .setPhone(payload.getPhone())
                        .setEmail(payload.getEmail())
                        .build())
                .setRideId(payload.getRide().getRideId());
        for (BusesPassenger passenger : payload.getRequestPassengers()) {
            requestBuilder.addPassengers(BusesModelsHelpers.busesPassengerToTPassenger(passenger));
        }
        TBookRequest request = requestBuilder.build();
        TBookResponse response;
        BusesService busesService = busesServiceProvider.getBusesServiceByOrderItem(orderItem);
        try {
            response = busesService.book(request, orderItem.getTestContext());
        } catch (Exception e) {
            if (e instanceof BusesServiceRetryableException &&
                    ctx.getAttempt() < busProperties.getReservationMaxTries()) {
                throw new RetryableException(e, busProperties.getReservationRetryDelay());
            }
            log.warn("Exception when booking: ", e);
            ctx.setState(EOrderItemState.IS_CANCELLED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), TServiceCancelled.newBuilder()
                    .setServiceId(orderItem.getId().toString()).build());
            return;
        }
        Preconditions.checkState(response.getOrder().getStatus() == EOrderStatus.OS_BOOKED,
                "Unexpected confirmed order state: %s", response.getOrder().getStatus());
        var feeMap = payload.getRide().getTicketTypeToFee();
        payload.setOrder(BusesModelsHelpers.tOrderToBusesOrder(response.getOrder(), feeMap));
        Preconditions.checkState(payload.getOrder().getExpiresAt() != null, "ExpiresAt can not be null after booking");
        orderItem.setExpiresAt(payload.getOrder().getExpiresAt());

        ctx.setState(EOrderItemState.IS_CALCULATING_FEE_BUS);
        ctx.scheduleEvent(TFeeCalculationCommit.newBuilder().build());
    }
}
