package ru.yandex.travel.orders.workflows.orderitem.bus.jobs;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.repository.BusOrderItemRepository;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@RequiredArgsConstructor
@Slf4j
public class BusOrderItemExpiredService {
    private final BusOrderItemRepository busOrderItemRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final List<EOrderItemState> orderItemStates = List.of(EOrderItemState.IS_RESERVED);

    @TransactionMandatory
    public List<UUID> fetchExpiredOrderItems(Set<UUID> active, int maxResultSize) {
        return busOrderItemRepository.getExpiredOrderItems(Instant.now(), orderItemStates, EWorkflowState.WS_RUNNING,
                active, PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long countExpiredOrderItems(Set<UUID> active) {
        return busOrderItemRepository.countExpiredOrderItems(Instant.now(), orderItemStates,
                EWorkflowState.WS_RUNNING, active);
    }

    @TransactionMandatory
    public void startCancellation(UUID orderItemId) {
        BusOrderItem orderItem = busOrderItemRepository.getOne(orderItemId);
        try (var ignored = NestedMdc.forEntity(orderItem.getId(), orderItem.getEntityType())) {
            if (orderItem.isExpired()) {
                log.info("Already expired, returning");
                return;
            }

            log.info("OrderItem {} expired at {}", orderItemId, orderItem.getExpiresAt());

            orderItem.setExpired(true);
            var message = TReservationExpired.newBuilder();
            workflowMessageSender.scheduleEvent(orderItem.getWorkflow().getId(), message.build());
        }
    }
}
