package ru.yandex.travel.orders.workflows.orderitem.bus.jobs;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(BusProperties.class)
public class BusOrderItemTaskProcessorsConfiguration {
    private final BusProperties busProperties;
    private final BusOrderItemExpiredService busOrderItemExpiredService;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    @Bean
    public TaskProcessor<UUID> busOrderItemCheckingExpirationProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return busOrderItemExpiredService.fetchExpiredOrderItems(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return busOrderItemExpiredService.countExpiredOrderItems(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("BusExpirationChecking");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, busOrderItemExpiredService::startCancellation,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("BusExpirationChecking")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(busProperties.getCheckExpirationTaskNumberOfItems())
                        .initialStartDelay(busProperties.getCheckExpirationTaskInitialStartDelay())
                        .scheduleRate(busProperties.getCheckExpirationTaskPeriod()).build());
    }
}
